package org.mule.weave.v2.module.excel

import org.apache.poi.openxml4j.util.ZipSecureFile
import org.apache.poi.ss.usermodel._
import org.apache.poi.ss.util.CellReference
import org.apache.poi.util.IOUtils
import org.mule.weave.v2.model.EvaluationContext
import org.mule.weave.v2.model.capabilities.EmptyLocationCapable
import org.mule.weave.v2.model.capabilities.UnknownLocationCapable
import org.mule.weave.v2.model.structure.ArraySeq
import org.mule.weave.v2.model.structure.KeyValuePair
import org.mule.weave.v2.model.structure.ObjectSeq
import org.mule.weave.v2.model.values._
import org.mule.weave.v2.module
import org.mule.weave.v2.module.reader.Reader
import org.mule.weave.v2.module.reader.SourceProvider
import org.mule.weave.v2.module.reader.SourceProviderAwareReader
import org.mule.weave.v2.module.excel.DefaultCellUtils._
import org.mule.weave.v2.module.excel.ExcelHelper.isEmptyRow
import org.mule.weave.v2.module.pojo.reader.ReflectionJavaValueConverter
import org.mule.weave.v2.parser.location.UnknownLocation

import java.io.File
import java.io.InputStream
import scala.collection.mutable.LinkedHashMap

class ExcelReader(override val sourceProvider: SourceProvider, val settings: ExcelReaderSettings)(implicit ctx: EvaluationContext) extends Reader with SourceProviderAwareReader {

  override def dataFormat: Option[module.DataFormat[_, _]] = Some(new ExcelDataFormat)

  private val inputStream = sourceProvider.asInputStream

  /**
    * Should return the root value of the document that was read
    *
    * @param name The name of the variable that is being referenced example payload
    * @return The root value
    */
  override protected def doRead(name: String): Value[_] = {
    if (settings.maxEntrySize.isDefined) {
      IOUtils.setByteArrayMaxOverride(settings.maxEntrySize.get)
    } else {
      IOUtils.setByteArrayMaxOverride(Integer.MAX_VALUE)
    }
    if (!settings.zipBombCheck) {
      ZipSecureFile.setMinInflateRatio(0)
    } else {
      if (settings.minInflateRatio.isDefined) {
        ZipSecureFile.setMinInflateRatio(settings.minInflateRatio.get)
      }
    }

    val wb = WorkbookFactory.create(inputStream)
    val numberOfSheets = wb.getNumberOfSheets
    val sheets = for (i <- 0 until numberOfSheets) yield {
      val sheet = wb.getSheetAt(i)
      KeyValuePair(KeyValue(sheet.getSheetName), new ExcelSheetValue(sheet, settings))
    }

    new MaterializedObjectValue(ObjectSeq(sheets), UnknownLocationCapable)
  }
}

object ExcelReader {

  def apply(file: File, settings: ExcelReaderSettings)(implicit ctx: EvaluationContext): ExcelReader = {
    new ExcelReader(SourceProvider(file), settings)
  }

  def apply(inputStream: InputStream, settings: ExcelReaderSettings)(implicit ctx: EvaluationContext): ExcelReader = {
    new ExcelReader(SourceProvider(inputStream), settings)
  }

  def apply(sourceProvider: SourceProvider, settings: ExcelReaderSettings)(implicit ctx: EvaluationContext): ExcelReader = {
    new ExcelReader(sourceProvider, settings)
  }

  def apply(content: String, settings: ExcelReaderSettings)(implicit ctx: EvaluationContext): ExcelReader = {
    new ExcelReader(SourceProvider(content), settings)
  }
}

class ExcelSheetValue(val sheet: Sheet, settings: ExcelReaderSettings)(implicit ctx: EvaluationContext) extends AlreadyMaterializedArrayValue with EmptyLocationCapable {

  def headerRowNum(): Int = {
    settings._headerRowNum = if (settings.tableOffset.isDefined) settings.headerRowNum() else sheet.getFirstRowNum
    settings.headerRowNum()
  }

  def headerColNum(row: Row): Int = {
    settings._headerColNum = if (settings.tableOffset.isDefined) settings.headerColNum() else row.getFirstCellNum.toInt
    settings.headerColNum()
  }

  val tableColNames: LinkedHashMap[Int, String] = {
    val map = new LinkedHashMap[Int, String]()
    if (settings.header) {
      val rowNum = headerRowNum()
      val headerRow = sheet.getRow(rowNum)
      if (headerRow != null) {
        for (i <- headerColNum(headerRow) until headerRow.getLastCellNum) {
          val cell = headerRow.getCell(i)
          if (cell != null) {
            map.+=((i, cellToString(cell)))
          }
        }
        map
      } else {
        map
      }
    } else {
      map
    }
  }

  override def evaluate(implicit ctx: EvaluationContext): T = {
    ArraySeq(new ExcelSheetArraySeq(sheet, settings, tableColNames))
  }
}

class ExcelSheetArraySeq(val sheet: Sheet, settings: ExcelReaderSettings, tableColNames: LinkedHashMap[Int, String])(implicit ctx: EvaluationContext) extends Iterator[Value[_]] {
  private var rowIndex = settings.bodyRowNum
  private var _hasNext = false
  private var _next: Option[Row] = None

  override def hasNext: Boolean = {
    if (!_hasNext) {
      while (rowIndex <= sheet.getLastRowNum) {
        val row: Row = sheet.getRow(rowIndex)
        val rowHasContent = !isEmptyRow(row)
        val keepEmptyLine = !settings.ignoreEmptyLine
        if (rowHasContent || keepEmptyLine) {
          _next = Option(row)
          _hasNext = true
          return true
        }
        rowIndex += 1
      }
      false
    } else {
      true
    }

  }

  override def next(): Value[_] = {
    if (hasNext) {
      new ExcelRowValue(fetchNext, settings, tableColNames)
    } else {
      throw new IndexOutOfBoundsException("No more rows.")
    }
  }

  private def fetchNext: Option[Row] = {
    _hasNext = false
    rowIndex += 1
    val result = _next
    _next = None
    result
  }
}

class ExcelRowValue(row: Option[Row], settings: ExcelReaderSettings, tableColNames: LinkedHashMap[Int, String])(implicit ctx: EvaluationContext) extends AlreadyMaterializedObjectValue with EmptyLocationCapable {

  var cachedValue: Seq[KeyValuePair] = _
  private val headerColNum = settings.headerColNum()

  def cellValues(): Seq[KeyValuePair] = {
    row
      .map((row) => {
        val lastColNum: Int = {
          settings.tableLimit match {
            case TableLimit.HEADER_SIZE => {
              headerColNum + tableColNames.size
            }
            case TableLimit.UNBOUNDED => {
              Math.max(headerColNum + tableColNames.size, row.getLastCellNum)
            }
            case colStr => {
              CellReference.convertColStringToIndex(colStr) + 1
            }
          }
        }
        getRowCellValues(row, headerColNum, lastColNum)
      })
      .getOrElse(createEmptyRow())
  }

  override def evaluate(implicit ctx: EvaluationContext): T = {
    if (cachedValue == null) {
      cachedValue = cellValues()
    }
    ObjectSeq(cachedValue)
  }

  def createEmptyRow(): Seq[KeyValuePair] = {
    if (settings.header) {
      tableColNames
        .flatMap((kv) => {
          val colName = KeyValue(kv._2)
          Seq(KeyValuePair(colName, StringValue("")))
        })
        .toSeq
    } else {
      Seq(KeyValuePair(KeyValue(""), StringValue("")))
    }

  }

  def getRowCellValues(row: Row, firstColNum: Int, lastCellNum: Int): Seq[KeyValuePair] = {
    val result = for (i <- firstColNum until lastCellNum) yield {
      val cell = row.getCell(i)
      if (settings.header && tableColNames.nonEmpty && i <= tableColNames.last._1 && tableColNames.isDefinedAt(i)) {
        val colName = KeyValue(tableColNames(i))
        val cellValue = readCell(cell)
        KeyValuePair(colName, cellValue)
      } else {
        val zeroBasedColIndex = i - firstColNum
        val colName = KeyValue(CellReference.convertNumToColString(zeroBasedColIndex))
        val cellValue = readCell(cell)
        KeyValuePair(colName, cellValue)
      }
    }
    result
  }

}

object DefaultCellUtils extends CellUtils {

  protected def formulaToString(cell: Cell)(implicit ctx: EvaluationContext): String = {
    doCellToString(cell, cell.getCachedFormulaResultType)
  }

  protected def formulaToValue(cell: Cell)(implicit ctx: EvaluationContext): Value[_] = {
    doReadCell(cell, cell.getCachedFormulaResultType)
  }
}

trait CellUtils {

  protected def formulaToString(cell: Cell)(implicit ctx: EvaluationContext): String
  protected def formulaToValue(cell: Cell)(implicit ctx: EvaluationContext): Value[_]

  def cellToString(cell: Cell)(implicit ctx: EvaluationContext): String = {
    doCellToString(cell, cell.getCellType)
  }

  def doCellToString(cell: Cell, cellType: CellType)(implicit ctx: EvaluationContext): String = {
    cellType match {
      case CellType.STRING => cell.getStringCellValue
      case CellType.NUMERIC if DateUtil.isCellDateFormatted(cell) => cell.getDateCellValue.toString
      case CellType.NUMERIC => cell.getNumericCellValue.toString
      case CellType.BOOLEAN => cell.getBooleanCellValue.toString
      case CellType.FORMULA => formulaToString(cell)
      case _ => ""
    }
  }

  def readCell(cell: Cell)(implicit ctx: EvaluationContext): Value[_] = {
    if (cell == null) {
      StringValue("")
    } else {
      doReadCell(cell, cell.getCellType)
    }
  }

  def doReadCell(cell: Cell, cellType: CellType)(implicit ctx: EvaluationContext): Value[_] = {
    cellType match {
      case CellType.STRING => StringValue(cell.getStringCellValue)
      case CellType.NUMERIC if DateUtil.isCellDateFormatted(cell) => ReflectionJavaValueConverter.convert(cell.getDateCellValue, () => UnknownLocation.locationString)
      case CellType.NUMERIC => NumberValue(cell.getNumericCellValue)
      case CellType.BOOLEAN => BooleanValue(cell.getBooleanCellValue)
      case CellType.FORMULA => formulaToValue(cell)
      case _ => StringValue("")
    }
  }
}

object ExcelHelper {
  def isEmptyRow(row: Row): Boolean = {
    row == null || row.getPhysicalNumberOfCells == 0
  }
}
