package org.mule.weave.v2.module.excel

import org.apache.poi.util.TempFileCreationStrategy
import org.mule.weave.v2.core.io.FileHelper
import org.mule.weave.v2.module.excel.ExcelCustomTempFileCreationStrategy.POI_FILES
import org.mule.weave.v2.module.reader.ResourceManager

import java.io.File
import java.io.IOException
import java.nio.file.Files
import java.nio.file.Paths
import java.util.concurrent.locks.Lock
import java.util.concurrent.locks.ReentrantLock

class ExcelCustomTempFileCreationStrategy extends TempFileCreationStrategy {
  private val dirLock: Lock = new ReentrantLock()
  private val resource = new ThreadLocal[ResourceManager]()
  private var poiFilesDir: File = _

  def setResourceManager(rm: ResourceManager): Unit = {
    resource.set(rm)
  }

  def cleanResourceManager(): Unit = {
    resource.remove()
  }

  override def createTempFile(prefix: String, suffix: String): File = {
    createPOIFilesDirectory()
    val newFile = Files.createTempFile(poiFilesDir.toPath, prefix, suffix).toFile
    deleteFileOnFinish(newFile)
    newFile
  }

  private def deleteFileOnFinish(newFile: File): Unit = {
    val manager = resource.get()
    if (manager != null) {
      manager.registerCloseable(new AutoCloseable {
        override def close(): Unit = {
          newFile.delete()
        }
      })
    }
  }

  private def createPOIFilesDirectory(): Unit = {
    // Create our temp dir only once by double-checked locking
    // The directory is not deleted, even if it was created by this TempFileCreationStrategy
    if (poiFilesDir == null || !poiFilesDir.exists()) {
      dirLock.lock()
      try {
        if (poiFilesDir == null || !poiFilesDir.exists()) {
          val tmpDir = FileHelper.tmpDir()
          if (tmpDir == null) {
            throw new IOException("System's temporary directory not defined - set the -Djava.io.tmpdir jvm property!")
          }
          val dirPath = Paths.get(tmpDir.getPath, POI_FILES)
          this.poiFilesDir = Files.createDirectories(dirPath).toFile
        }
      } finally {
        dirLock.unlock()
      }
    }
  }

  override def createTempDirectory(prefix: String): File = {
    createPOIFilesDirectory()
    val newDirectory = Files.createTempDirectory(poiFilesDir.toPath, prefix).toFile
    newDirectory
  }
}

object ExcelCustomTempFileCreationStrategy {
  val POI_FILES: String = "poifiles"
}
