package org.mule.weave.v2.module.excel

import org.apache.commons.compress.archivers.zip.Zip64Mode
import org.apache.poi.openxml4j.util.ZipSecureFile
import org.apache.poi.ss.usermodel.DateUtil
import org.apache.poi.util.TempFile
import org.apache.poi.xssf.streaming.SXSSFCell
import org.apache.poi.xssf.streaming.SXSSFRow
import org.apache.poi.xssf.streaming.SXSSFSheet
import org.apache.poi.xssf.streaming.SXSSFWorkbook
import org.mule.weave.v2.model.EvaluationContext
import org.mule.weave.v2.model.structure.ObjectSeq
import org.mule.weave.v2.model.types.DateTimeType
import org.mule.weave.v2.model.types._
import org.mule.weave.v2.model.values.Value
import org.mule.weave.v2.model.values.math.Number
import org.mule.weave.v2.module.DataFormat
import org.mule.weave.v2.module.commons.java.writer.converter.DataConverter
import org.mule.weave.v2.module.excel.ExcelWriter.TEMP_FILE_CREATION_STRATEGY
import org.mule.weave.v2.module.pojo.writer.converter.JavaDataConverter.CalendarDataConverter$
import org.mule.weave.v2.module.writer.TargetProvider
import org.mule.weave.v2.module.writer.Writer

import java.io.OutputStream
import java.time.format.DateTimeFormatter
import java.util.Calendar

class ExcelWriter(outputStream: OutputStream, val settings: ExcelSettings)(implicit ctx: EvaluationContext) extends Writer {

  private lazy val wb = {
    val workbook: SXSSFWorkbook = new SXSSFWorkbook(1000)
    workbook.setCompressTempFiles(true)
    workbook.setZip64Mode(Zip64Mode.AsNeeded)
    workbook
  }

  override def result: Any = outputStream

  protected override def doWriteValue(value: Value[_])(implicit ctx: EvaluationContext): Unit = {
    TEMP_FILE_CREATION_STRATEGY.setResourceManager(ctx.serviceManager.resourceManager)
    try {
      if (!settings.zipBombCheck) {
        ZipSecureFile.setMinInflateRatio(0)
      } else {
        if (settings.minInflateRatio.isDefined) {
          ZipSecureFile.setMinInflateRatio(settings.minInflateRatio.get)
        }
      }
      val objectSeq = ObjectType.coerce(value).evaluate.toIterator()
      for (keyValuePair <- objectSeq) {
        val sheetName = keyValuePair._1.evaluate.name
        val sheetContent = ArrayType.coerce(keyValuePair._2).evaluate
        writeSheet(sheetName, sheetContent.toIterator())
      }

      wb.write(outputStream)
    } finally {
      TEMP_FILE_CREATION_STRATEGY.cleanResourceManager()
    }
  }

  def writeSheet(sheetName: String, sheetContent: Iterator[Value[_]])(implicit ctx: EvaluationContext): Unit = {
    val sheet = wb.createSheet(sheetName)
    var rowNum = settings.headerRowNum()

    if (settings.header && sheetContent.hasNext) {
      val rowValue = sheetContent.next()
      val rowObjectSeq = ObjectType.coerce(rowValue).evaluate
      doWriteFirstRow(sheet, rowObjectSeq, rowNum)
      rowNum += 2
    }

    while (sheetContent.hasNext) {
      val row = sheet.createRow(rowNum)
      val rowValue = sheetContent.next()
      val rowObjectSeq = ObjectType.coerce(rowValue).evaluate
      doWriteRow(row, rowObjectSeq)
      rowNum += 1
    }
  }

  def doWriteFirstRow(sheet: SXSSFSheet, rowValue: ObjectSeq, rowNum: Int)(implicit ctx: EvaluationContext): Unit = {
    val row0 = sheet.createRow(rowNum)
    val row1 = sheet.createRow(rowNum + 1)

    var colNumber = settings.headerColNum()
    val keyValuePairs = rowValue.toIterator()
    while (keyValuePairs.hasNext) {
      val keyValuePair = keyValuePairs.next()
      val headerCell = row0.createCell(colNumber)
      val headerString = KeyType.coerce(keyValuePair._1).evaluate.name
      headerCell.setCellValue(headerString)
      val cell = row1.createCell(colNumber)
      writeCell(cell, keyValuePair._2)
      colNumber += 1
    }
  }

  def doWriteRow(row: SXSSFRow, rowValue: ObjectSeq)(implicit ctx: EvaluationContext): Unit = {
    var colNumber = settings.headerColNum()
    val keyValuePairs = rowValue.toIterator()
    while (keyValuePairs.hasNext) {
      val cell = row.createCell(colNumber)
      val cellValue = keyValuePairs.next()._2
      writeCell(cell, cellValue)
      colNumber += 1
    }

  }

  def writeCell(cell: SXSSFCell, cellValue: Value[_])(implicit ctx: EvaluationContext): Unit = {
    cellValue match {
      case v if StringType.accepts(v) => {
        val cellStr: String = StringType.coerce(v).evaluate.toString
        cell.setCellValue(cellStr)
      }
      case v if BooleanType.accepts(v) => {
        val booleanValue = BooleanType.coerce(v).evaluate
        cell.setCellValue(booleanValue)
      }
      case v if NumberType.accepts(v) => {
        val cellNumber: Number = NumberType.coerce(v).evaluate
        cell.setCellValue(cellNumber.toDouble)
      }
      case v if TimeType.accepts(v) => {
        val timeValue = TimeType.coerce(v).evaluate
        val timeStr = timeValue.format(DateTimeFormatter.ofPattern("HH:mm:ss"))
        val excelTime = DateUtil.convertTime(timeStr)
        cell.setCellValue(excelTime)
        setCellDataFormat(cell, "hh:mm:ss")
      }
      case v if DateTimeType.accepts(v) || LocalDateTimeType.accepts(v) => {
        val dateTime = DateTimeType.coerce(v).evaluate
        val calendar = CalendarDataConverter$.asCalendar(dateTime)
        setCellDataFormat(cell, "yyyy-MM-ddThh:mm:ss")
        cell.setCellValue(calendar)
      }
      case v if LocalDateType.accepts(v) => {
        val localDate = LocalDateType.coerce(v).evaluate
        val date = DataConverter.to[Calendar](localDate, v.schema).get.getTime
        cell.setCellValue(date)
        setCellDataFormat(cell, "yyyy-MM-dd")
      }
      case _ => //do nothing
    }
  }

  def setCellDataFormat(cell: SXSSFCell, format: String): Unit = {
    val cellStyle = wb.createCellStyle()
    cellStyle.setDataFormat(wb.getCreationHelper.createDataFormat().getFormat(format))
    cell.setCellStyle(cellStyle)
  }

  override def close(): Unit = {
    wb.dispose()
    outputStream.close()
  }

  override def flush(): Unit = {
    outputStream.flush()
  }

  override def dataFormat: Option[DataFormat[_, _]] = Some(new ExcelDataFormat)
}

object ExcelWriter {

  val TEMP_FILE_CREATION_STRATEGY: ExcelCustomTempFileCreationStrategy = initTempStrategy()

  private def initTempStrategy() = {
    val strategy = new ExcelCustomTempFileCreationStrategy()
    TempFile.setTempFileCreationStrategy(strategy)
    strategy
  }

  def apply(tp: TargetProvider, settings: ExcelSettings)(implicit ctx: EvaluationContext): ExcelWriter = {
    new ExcelWriter(tp.asOutputStream, settings)
  }
}
