package org.mule.weave.v2.module.flatfile

import java.io.OutputStream
import java.{ util => ju }

import com.mulesoft.flatfile.schema.fixedwidth.FlatFileSegmentWriter
import com.mulesoft.flatfile.schema.fixedwidth.FlatFileStructureWriter
import com.mulesoft.flatfile.schema.fixedwidth.FlatFileWriterBase
import com.mulesoft.flatfile.schema.fixedwidth.FlatFileWriterConfig
import com.mulesoft.flatfile.schema.model.Copybook
import com.mulesoft.flatfile.schema.model.EdiSchemaVersion
import com.mulesoft.flatfile.schema.model.Structure
import org.mule.weave.v2.core.exception.WriterExecutionException
import org.mule.weave.v2.model.EvaluationContext
import org.mule.weave.v2.model.types.ArrayType
import org.mule.weave.v2.model.types.ObjectType
import org.mule.weave.v2.model.values.Value
import org.mule.weave.v2.module.DataFormat
import org.mule.weave.v2.module.writer.TargetProvider
import org.mule.weave.v2.module.writer.Writer
import org.mule.weave.v2.module.flatfile.output.ConvertValue

class FlatFileWriter(os: OutputStream, val settings: FlatFileWriterSettings, structOverride: Option[Structure] = None)(implicit ctx: EvaluationContext) extends Writer {

  /** Build writer configuration based on settings and schema defaults. */
  private def buildConfig(version: EdiSchemaVersion): FlatFileWriterConfig = {
    val defaultChar: Int = version.ediForm match {
      case Copybook => 0
      case _        => '0'
    }
    val missingChar = settings.missingValueCharacter(defaultChar)
    val expectedBehavior = settings.missingValueExpectedBehavior()
    FlatFileWriterConfig(true, settings.charset, settings.recordTerminatorString, missingChar.toChar,
      settings.trimValues, settings.zonedDecimalStrict, !settings.truncateDependingOn, expectedBehavior,
      settings.notTruncateDependingOnSubjectNotPresent, settings.useMissCharAsDefaultForFill, settings.fillRedefinesByMaxLength)
  }

  lazy val ffWriter: FlatFileWriterBase = {
    structOverride match {
      case Some(s) => new FlatFileStructureWriter(os, s, buildConfig(s.version))
      case None =>
        val schema = settings.loadedSchema
        val config = buildConfig(schema.ediVersion)
        settings.getStructure(schema) match {
          case Some(s) => new FlatFileStructureWriter(os, s, config)
          case None =>
            settings.getSegment(schema) match {
              case Some(s) => new FlatFileSegmentWriter(os, s, config)
              case None    => throw new IllegalStateException("Need to specify structureIdent or schemaIdent in writer configuration")
            }
        }
    }
  }

  override def result: Any = os

  private def write(data: Object): Unit = {
    val wrapper = new ju.HashMap[String, Object]
    wrapper put ("Data", data)
    ffWriter.write(wrapper).get
  }

  protected override def doWriteValue(v: Value[_])(implicit ctx: EvaluationContext): Unit = {
    v match {
      case objectValue if ObjectType.accepts(v) => write(ConvertValue.convertValue(ObjectType.coerce(objectValue)))
      case arrayValue if ArrayType.accepts(v)   => write(ConvertValue.convertValue(ArrayType.coerce(arrayValue)))
      case x                                    => throw new WriterExecutionException(v.location(), getName(), s"Fixed width data should be an Array<Object> or Object but got ${v.valueType}")
    }
  }

  override def close() {}

  override def dataFormat: Option[DataFormat[_, _]] = Some(new FlatFileDataFormat)
}

object FlatFileWriter {
  def apply(tp: TargetProvider, settings: FlatFileWriterSettings)(implicit ctx: EvaluationContext): FlatFileWriter = {
    new FlatFileWriter(tp.asOutputStream, settings)
  }

  def apply(os: OutputStream, settings: FlatFileWriterSettings, structOverride: Option[Structure] = None)(implicit ctx: EvaluationContext): FlatFileWriter = {
    new FlatFileWriter(os, settings, structOverride)
  }
}
