package org.mule.weave.v2.module.flatfile

import com.mulesoft.flatfile.schema.model.EdiForm
import com.mulesoft.flatfile.schema.model.EdiSchema
import com.mulesoft.flatfile.schema.model.Segment
import com.mulesoft.flatfile.schema.model.Structure
import com.mulesoft.flatfile.schema.yaml.YamlReader
import org.mule.weave.v2.core.exception.InvalidOptionValueException
import org.mule.weave.v2.core.io.SeekableStream
import org.mule.weave.v2.model.EvaluationContext
import org.mule.weave.v2.module.DataFormat
import org.mule.weave.v2.module.option.BooleanModuleOption
import org.mule.weave.v2.module.option.ModuleOption
import org.mule.weave.v2.module.option.OptionKind
import org.mule.weave.v2.module.option.OptionalStringModuleOption
import org.mule.weave.v2.module.option.SchemaStreamAwareSetting
import org.mule.weave.v2.module.option.Settings
import org.mule.weave.v2.module.reader.ConfigurableDeferred
import org.mule.weave.v2.module.writer.ConfigurableEncoding
import org.mule.weave.v2.parser.location.UnknownLocation

import java.io.File
import java.io.FileReader
import java.io.InputStreamReader
import java.nio.charset.Charset
import java.nio.file.Files
import java.util
import java.util.concurrent.ConcurrentHashMap

abstract class FlatFileSettings extends Settings with SchemaStreamAwareSetting {

  var schemaPath: Option[String] = None
  var structureIdent: Option[String] = None
  var segmentIdent: Option[String] = None
  var enforceRequires: Boolean = false
  var missingValues: Option[String] = None
  var zonedDecimalStrict: Boolean = false
  var truncateDependingOn: Boolean = false
  var notTruncateDependingOnSubjectNotPresent: Boolean = false
  var useMissCharAsDefaultForFill: Boolean = false

  val MISSING_VALUES_NONE: String = "none"
  val MISSING_VALUES_ZEROES: String = "zeroes"
  val MISSING_VALUES_SPACES: String = "spaces"
  val MISSING_VALUES_NULLS: String = "nulls"

  def missingValueCharacter(dflt: Int): Int = {
    if (missingValues.isEmpty) dflt
    else {
      missingValueExpectedBehavior()
      missingValues.get.toLowerCase match {
        case MISSING_VALUES_SPACES => ' '
        case MISSING_VALUES_ZEROES => '0'
        case MISSING_VALUES_NULLS  => 0
        case MISSING_VALUES_NONE   => -1
        case _ =>
          throw new InvalidOptionValueException(UnknownLocation, optionName = "missingValues", validValues = Some(missingValueChoices))
      }
    }
  }

  def missingValueExpectedBehavior(): Boolean = {
    if (missingValues.isEmpty) false
    else if (missingValues.get.equals(missingValues.get.toLowerCase())) false
    else if (missingValues.get.equals(missingValues.get.toUpperCase)) true
    else
      throw new InvalidOptionValueException(UnknownLocation, optionName = "missingValues", validValues = Some(missingValueChoices))
  }

  def missingValueChoices = List(MISSING_VALUES_NONE, MISSING_VALUES_SPACES, MISSING_VALUES_ZEROES, MISSING_VALUES_NULLS)

  override def loadSettingsOptions(): Map[String, ModuleOption] = {
    super.loadSettingsOptions ++
      Map(
        OptionalStringModuleOption(name = "schemaPath", descriptionUrl = "data-format/flatfile/schemaPath.asciidoc", required = true),
        OptionalStringModuleOption(name = "structureIdent", descriptionUrl = "data-format/flatfile/structureIdent.asciidoc"),
        OptionalStringModuleOption(name = "segmentIdent", descriptionUrl = "data-format/flatfile/segmentIdent.asciidoc"),
        BooleanModuleOption(name = "enforceRequires", descriptionUrl = "data-format/flatfile/enforceRequires.asciidoc", defaultValue = false),
        OptionalStringModuleOption(name = "missingValues", descriptionUrl = "data-format/flatfile/missingValues.asciidoc", possibleValues = missingValueChoices.toSet),
        BooleanModuleOption(name = "zonedDecimalStrict", descriptionUrl = "data-format/flatfile/zonedDecimalStrict.asciidoc", defaultValue = false),
        BooleanModuleOption(name = "truncateDependingOn", descriptionUrl = "data-format/flatfile/truncateDependingOn.asciidoc", defaultValue = false),
        BooleanModuleOption(name = "notTruncateDependingOnSubjectNotPresent", descriptionUrl = "data-format/flatfile/notTruncateDependingOnSubjectNotPresent.asciidoc", defaultValue = false),
        BooleanModuleOption(name = "useMissCharAsDefaultForFill", descriptionUrl = "data-format/flatfile/useMissCharAsDefaultForFill.asciidoc", defaultValue = false))
  }

  protected override def writeSettingsValue(settingName: String, value: Any): Unit = {
    settingName match {
      case "schemaPath" => {
        this.schemaPath = value.asInstanceOf[Option[String]]
      }
      case "structureIdent" => {
        this.structureIdent = value.asInstanceOf[Option[String]]
      }
      case "segmentIdent" => {
        this.segmentIdent = value.asInstanceOf[Option[String]]
      }
      case "enforceRequires" => {
        this.enforceRequires = value.asInstanceOf[Boolean]
      }
      case "missingValues" => {
        this.missingValues = value.asInstanceOf[Option[String]]
      }
      case "zonedDecimalStrict" => {
        this.zonedDecimalStrict = value.asInstanceOf[Boolean]
      }
      case "truncateDependingOn" => {
        this.truncateDependingOn = value.asInstanceOf[Boolean]
      }
      case "notTruncateDependingOnSubjectNotPresent" => {
        this.notTruncateDependingOnSubjectNotPresent = value.asInstanceOf[Boolean]
      }
      case "useMissCharAsDefaultForFill" => {
        this.useMissCharAsDefaultForFill = value.asInstanceOf[Boolean]
      }
      case _ => super.writeSettingsValue(settingName, value)
    }
  }

  lazy val loadedSchema: EdiSchema = schemaPath match {
    case Some(path) => SchemaCache.loadCached(path)
    case None       => throw new IllegalStateException("No schema set")
  }

  /** Get the referenced structure definition from schema. */
  def getStructure(schema: EdiSchema): Option[Structure] = {
    val structs = schema.structures
    structureIdent match {
      case Some(ident) =>
        structs.get(ident) match {
          case s: Some[Structure] => s
          case None               => throw new IllegalStateException(s"Structure $ident not defined in schema")
        }
      case None =>
        if (structs.isEmpty) None
        else if (structs.size == 1) Some(structs.head._2)
        else throw new IllegalStateException("Structure identifier must be set for schema with multiple structures")
    }
  }

  /** Get the referenced segment definition from schema. */
  def getSegment(schema: EdiSchema): Option[Segment] = {
    val segs = schema.segments
    segmentIdent match {
      case Some(ident) =>
        segs.get(ident) match {
          case s: Some[Segment] => s
          case None             => throw new IllegalStateException(s"Segment $ident not defined in schema")
        }
      case None =>
        if (segs.size == 1) Some(segs.head._2)
        else None
    }
  }

  def dumpFlatFileSchema(dumpFolder: File, ctx: EvaluationContext, schemaName: String): Unit = {
    val schemaStream = this.schemaStream()
    if (schemaStream.isDefined) {
      val dumpSchemaFile = new File(dumpFolder, schemaName)
      val content = SeekableStream(schemaStream.get)(ctx)
      try {
        content.resetStream()
        Files.copy(content, dumpSchemaFile.toPath)
      } finally {
        content.close()
      }
    }
  }

}

class FlatFileWriterSettings(val dataFormat: DataFormat[_, _]) extends FlatFileSettings with ConfigurableEncoding with ConfigurableDeferred {

  var recordTerminator: Option[String] = None
  var trimValues: Boolean = false
  var fillRedefinesByMaxLength: Boolean = false

  val RECORD_TERMINATOR_LF = "lf"
  val RECORD_TERMINATOR_CR = "cr"
  val RECORD_TERMINATOR_CRLF = "crlf"
  val RECORD_TERMINATOR_NONE = "none"

  def recordTerminatorChoices = List(RECORD_TERMINATOR_LF, RECORD_TERMINATOR_CR, RECORD_TERMINATOR_CRLF, RECORD_TERMINATOR_NONE)

  def recordTerminatorString: String = {
    if (recordTerminator.isEmpty) {
      System.getProperty("line.separator")
    } else {
      recordTerminator.get match {
        case RECORD_TERMINATOR_LF   => "\n"
        case RECORD_TERMINATOR_CR   => "\r"
        case RECORD_TERMINATOR_CRLF => "\r\n"
        case RECORD_TERMINATOR_NONE => ""
        case _ =>
          throw new InvalidOptionValueException(UnknownLocation, optionName = "recordTerminator", validValues = Some(recordTerminatorChoices))
      }
    }
  }

  override def loadSettingsOptions(): Map[String, ModuleOption] = {
    super.loadSettingsOptions ++ Map(
      OptionalStringModuleOption(name = "recordTerminator", descriptionUrl = "data-format/flatfile/recordTerminator.asciidoc", possibleValues = recordTerminatorChoices.toSet, kind = OptionKind.WRITER),
      BooleanModuleOption(name = "trimValues", descriptionUrl = "data-format/flatfile/trimValues.asciidoc", defaultValue = false, kind = OptionKind.WRITER),
      BooleanModuleOption(name = "fillRedefinesByMaxLength", descriptionUrl = "data-format/flatfile/fillRedefinesByMaxLength.asciidoc", defaultValue = false, kind = OptionKind.WRITER))
  }

  protected override def writeSettingsValue(settingName: String, value: Any): Unit = {
    settingName match {
      case "recordTerminator" => {
        this.recordTerminator = value.asInstanceOf[Option[String]]
      }
      case "trimValues" => {
        this.trimValues = value.asInstanceOf[Boolean]
      }
      case "fillRedefinesByMaxLength" => {
        this.fillRedefinesByMaxLength = value.asInstanceOf[Boolean]
      }
      case _ => super.writeSettingsValue(settingName, value)
    }
  }
}

class FlatFileReaderSettings extends FlatFileSettings {

  var recordParsing: Option[String] = None
  var allowLenientWithBinaryNotEndElement: Boolean = false
  var retainEmptyStringFieldsOnParsing: Boolean = false
  var substituteCharacterAsMissingValue: Boolean = false

  val RECORD_PARSING_STRICT = "strict"
  val RECORD_PARSING_LENIENT = "lenient"
  val RECORD_PARSING_NO_TERMINATOR = "noTerminator"
  val RECORD_PARSING_SINGLE_RECORD = "singleRecord"

  def recordParsingChoices = List(RECORD_PARSING_STRICT, RECORD_PARSING_LENIENT, RECORD_PARSING_NO_TERMINATOR, RECORD_PARSING_SINGLE_RECORD)

  /** Returns (has terminator, lenient handling) */
  def recordParsingFlags: (Boolean, Boolean) =
    if (recordParsing.isEmpty) (true, false)
    else
      recordParsing.get match {
        case RECORD_PARSING_STRICT        => (true, false)
        case RECORD_PARSING_LENIENT       => (true, true)
        case RECORD_PARSING_NO_TERMINATOR => (false, false)
        case RECORD_PARSING_SINGLE_RECORD => (false, true)
        case _ =>
          throw new InvalidOptionValueException(UnknownLocation, optionName = "recordParsing", validValues = Some(recordParsingChoices))
      }

  override def loadSettingsOptions(): Map[String, ModuleOption] = {
    super.loadSettingsOptions +
      OptionalStringModuleOption(name = "recordParsing", defaultValue = Some("strict"), descriptionUrl = "data-format/flatfile/recordParsing.asciidoc", possibleValues = recordParsingChoices.toSet, kind = OptionKind.READER) +
      BooleanModuleOption(name = "allowLenientWithBinaryNotEndElement", descriptionUrl = "data-format/flatfile/allowLenientWithBinaryNotEndElement.asciidoc", defaultValue = false, kind = OptionKind.READER) +
      BooleanModuleOption(name = "retainEmptyStringFieldsOnParsing", descriptionUrl = "data-format/flatfile/retainEmptyStringFieldsOnParsing.asciidoc", defaultValue = false, kind = OptionKind.READER) +
      BooleanModuleOption(name = "substituteCharacterAsMissingValue", descriptionUrl = "data-format/flatfile/substituteCharacterAsMissingValue.asciidoc", defaultValue = false, kind = OptionKind.READER)
  }

  protected override def writeSettingsValue(settingName: String, value: Any): Unit = {
    settingName match {
      case "recordParsing" => {
        this.recordParsing = value.asInstanceOf[Option[String]]
      }
      case "allowLenientWithBinaryNotEndElement" => {
        this.allowLenientWithBinaryNotEndElement = value.asInstanceOf[Boolean]
      }
      case "retainEmptyStringFieldsOnParsing" => {
        this.retainEmptyStringFieldsOnParsing = value.asInstanceOf[Boolean]
      }
      case "substituteCharacterAsMissingValue" => {
        this.substituteCharacterAsMissingValue = value.asInstanceOf[Boolean]
      }
      case _ => super.writeSettingsValue(settingName, value)
    }
  }
}

object SchemaCache {

  val schemaFileCache = new ConcurrentHashMap[String, (Long, EdiSchema)]

  val contextMapCache = new util.WeakHashMap[ClassLoader, ConcurrentHashMap[String, EdiSchema]]()

  def loadCachedFileSchema(file: File): EdiSchema = {
    val abspath = file.getAbsolutePath

    def fileToCache: EdiSchema = {
      val modified = file.lastModified
      val reader = new FileReader(file)
      try {
        val yamlReader = new YamlReader(EdiForm)
        val schema = yamlReader.loadYaml(reader, Nil)
        schemaFileCache.put(abspath, (modified, schema))
        schema
      } finally {
        reader.close()
      }
    }

    if (schemaFileCache.containsKey(abspath)) {
      val (modified, schema) = schemaFileCache.get(abspath)
      if (modified >= file.lastModified) schema
      else {
        schemaFileCache.remove(abspath)
        fileToCache
      }
    } else fileToCache
  }

  def loadCachedClasspathSchema(path: String, ctx: ClassLoader, map: ConcurrentHashMap[String, EdiSchema]): EdiSchema = {
    if (map.containsKey(path)) map.get(path)
    else {
      val is = {
        val is0 = getClass.getResourceAsStream(path)
        if (is0 == null) {
          val relpath = if (path.startsWith("/")) path.substring(1) else path
          val is1 = getClass.getClassLoader.getResourceAsStream(relpath)
          if (is1 == null) {
            val is2 = Thread.currentThread.getContextClassLoader.getResourceAsStream(relpath)
            if (is2 == null) throw new IllegalArgumentException(s"file $path not found on any classpath")
            else is2
          } else is1
        } else is0
      }
      val reader = new InputStreamReader(is, Charset.forName("UTF-8"))
      try {
        val schema = new YamlReader(EdiForm).loadYaml(reader, Nil)
        map.put(path, schema)
        schema
      } finally {
        reader.close()
      }
    }
  }

  def loadCached(path: String): EdiSchema = {
    val file = new File(path)
    if (file.exists) loadCachedFileSchema(file)
    else {
      val context = Thread.currentThread.getContextClassLoader
      val ctxmap = contextMapCache.synchronized({
        if (contextMapCache.containsKey(context)) {
          contextMapCache.get(context)
        } else {
          val newmap = new ConcurrentHashMap[String, EdiSchema]
          contextMapCache.put(context, newmap)
          newmap
        }
      })
      loadCachedClasspathSchema(path, context, ctxmap)
    }
  }
}
