type HttpResponse = {
  headers?: { _?: String},
  body?: Any,
  status?: Number
}

type HttpRequest = {
  headers: {},
  method: String,
  path: String,
  body: Binary
}

type HttpHandler = (HttpRequest) -> HttpResponse

type HttpServerConfig = {
  port: Number,
  host: String,
  contentType?: String
}

type HttpServer = {
  running: Boolean,
  port: Number,
  host: String,
  stop: () -> Boolean
}

type APIDefinition = {
    _ ?: {
        GET?: HttpHandler,
        POST?: HttpHandler,
        PUT?: HttpHandler,
        DELETE?: HttpHandler,
        PATCH?: HttpHandler,
        OPTIONS?: HttpHandler,
        HEAD?: HttpHandler,
        TRACE?: HttpHandler
    }
}

/**
* Starts an http server at with the specified config.
* The handler with attend all the requests.
* Returns true if the server was initialized correctly
*/
fun server(configuration: HttpServerConfig, handler: HttpHandler): HttpServer = native("http::HttpServerFunction")

type HttpServerMiddleware = (req: HttpRequest, next: (ctx: HttpRequest) -> HttpResponse) -> HttpResponse

fun handleWithMiddleware(req: HttpRequest, tail: Array<HttpServerMiddleware>): HttpResponse =
  tail match {
    case [] -> { status: 501 }
    case [head ~ tail] ->
      head(
        req,
        (newCtx) -> handleWithMiddleware(newCtx, tail)
      )
  }

fun middlewareServer(config: HttpServerConfig = {port: 8081, host:"localhost"}, middlewares: Array<HttpServerMiddleware>): HttpServer =
  server(config, (request) ->
    using(res = handleWithMiddleware(request, middlewares))
    using(headers =
      (
        if(res.body? and (not res.headers."Content-Type"?))
          res.headers ++ {"Content-Type" : config.contentType default "application/json"}
        else
          res.headers
      ) default (
        if(res.body?)
          {"Content-Type" : config.contentType default "application/json"}
        else
          {}
      )
    )
      {
          //If there is body and not Content-Type header is defined use the one form the config
        headers: headers,
        (body: write(res.body, headers["Content-Type"])) if res.body?,
        status: res.status default 200
      }
  )

/**
* Initialize an api with the specified APIDefinition
*/
fun api(config: HttpServerConfig = {port: 8081, host:"localhost"}, handler: APIDefinition): HttpServer =
  server(config,
    (request) -> (
      using(methodHandler = handler[(request.path)][(request.method)])
        if(methodHandler != null)
          using(response = methodHandler(request), headers = response.headers default {})
            {
                //If there is body and not Content-Type header is defined use the one form the config
              headers: if(response.body? and (not headers."Content-Type"?))
                         headers ++ {"Content-Type" : config."Content-Type" default "application/json"}
                       else headers,
              (body: response.body) if response.body?,
              responseCode: response.responseCode default 200
            }
        else
          {
            body: "$(request.path) Not Found",
            responseCode: 404
          }
        )
  )
