import * from dw::core::Objects

type HttpResponse = {
  headers?: { _?: String},
  body?: Any,
  status?: Number
}

type HttpRequest = {
  headers: { _?: String },
  method: String,
  path: String,
  body: Binary
}

type HttpHandler = (HttpRequest) -> HttpResponse

type HttpServerConfig = {
  port: Number,
  host: String,
  contentType?: String
}

type HttpServer = {|
  running: Boolean,
  port: Number,
  host: String,
  stop: () -> Boolean
|}

type APIDefinition = {
    _ ?: {
        GET?: HttpHandler,
        POST?: HttpHandler,
        PUT?: HttpHandler,
        DELETE?: HttpHandler,
        PATCH?: HttpHandler,
        OPTIONS?: HttpHandler,
        HEAD?: HttpHandler,
        TRACE?: HttpHandler
    }
}

type InterseptorCapable = { Interceptors?:  Array<HttpInterseptor> }

type ApiConfig = HttpServerConfig & InterseptorCapable

type InterceptedHttpRequest = {
    /**
    * The chain of interseptor will be cut only if response is present
    */
    response?: HttpResponse,
    request?: HttpRequest
}

type HttpInterseptor = {onRequest?: (HttpRequest) -> InterceptedHttpRequest, onResponse?: (HttpRequest, HttpResponse) -> HttpResponse  }

fun isOptions(request: HttpRequest) = request.method == "OPTIONS"

//TODO improve with cors config for now use it hardcoded
fun CORS() =
  {
    onRequest: (
      (req) ->
        if(isOptions(req) and req.headers.Origin?)
          {
            response: {
              headers: {
                "Access-Control-Allow-Origin": "*",
                "Access-Control-Allow-Methods": "POST, GET, OPTIONS, DELETE, PUT",
                "Access-Control-Allow-Headers": "Origin, X-Requested-With, Content-Type, Accept, Accept-Encoding, Accept-Language, Host, Referer, User-Agent",
                "Access-Control-Max-Age": 60 * 60 * 24 * 20 // cache pre-flight response for 20 days
              }
            }
          }
        else
         {request: req}
      ),
    onResponse:
      ((req, resp) ->
        if(req.headers.Origin?)
          resp mergeWith { headers: resp.headers mergeWith {"Access-Control-Allow-Origin": "*"} }
        else
          resp
      )
  }

/**
* Starts an http server at with the specified config.
* The handler with attend all the requests.
* Returns true if the server was initialized correctly
*/
fun server(configuration: HttpServerConfig, handler: HttpHandler): HttpServer = native("http::HttpServerFunction")

fun handleRequestInterceptors(req: HttpRequest, interceptors: Array<HttpInterseptor>): InterceptedHttpRequest =
  interceptors match {
    case [] -> { request: req }
    case [head ~ tail] ->
      if(head.onRequest?)
        using(result = head.onRequest(req))
          if(result.response?)
            result
          else
            handleRequestInterceptors(result.request!, tail)
      else
        handleRequestInterceptors(req, tail)
  }

fun handleResponseInterceptors(req: HttpRequest,resp: HttpResponse, interceptors: Array<HttpInterseptor>): HttpResponse =
  interceptors match {
    case [] -> resp
    case [head ~ tail] ->
     if(head.onResponse?)
       handleResponseInterceptors(req, head.onResponse(req, resp), tail)
     else
       handleResponseInterceptors(req, resp, tail)
  }



/**
* Initialize an api with the specified APIDefinition
*/
fun api(config: ApiConfig = {port: 8081, host:"localhost"}, apiDefinition: APIDefinition): HttpServer =
  server(config,
    (request) ->
      using(
        matchingHandler = (apiDefinition[?(request.path matches ($$ as String))] pluck $)[0],
        methodHandler = matchingHandler[(request.method)],
        interseptedRequest =  handleRequestInterceptors(request, config.interceptors default [])
      )
        if(methodHandler != null)
          if(interseptedRequest.response?)
            interseptedRequest.response!
          else
            using(
              response = methodHandler(interseptedRequest.request!),
              headers = dw::http::BodyUtils::normalizeHeaders(response.headers default {})
             )
              handleResponseInterceptors(
                interseptedRequest.request!,
                {
                    //If there is body and not Content-Type header is defined use the one form the config
                  headers: if(response.body? and (not headers."Content-Type"?))
                           headers ++ {"Content-Type" : config."Content-Type" default "application/json"}
                         else headers,
                  (body: response.body) if response.body?,
                  responseCode: response.responseCode default 200
                }, config.interceptors default [])
        else
          {
            body: "$(request.path) Not Found",
            responseCode: 404
          }

  )

/**
* Returns a static resoure respone using the specified path. Use this method to server static content
*/
fun serveResource(path: String): HttpResponse = native("http::ServeResourceFunction")