package org.mule.weave.v2.module.pojo.reader

import org.mule.weave.v2.model.EvaluationContext
import org.mule.weave.v2.model.structure._
import org.mule.weave.v2.model.values._
import org.mule.weave.v2.module.commons.java.JavaValueHelper
import org.mule.weave.v2.module.commons.java.value.JavaLocation
import org.mule.weave.v2.module.commons.java.value.JavaObjectValue
import org.mule.weave.v2.module.pojo.exception.InvalidPropertyNameException

import scala.collection.mutable

class JavaBeanObjectValue(val bean: Any, val locationString: () => String) extends JavaObjectValue {
  private var materializationRequested: Boolean = false
  private val clazz = bean.getClass
  private val cache: mutable.Map[String, Value[_]] = mutable.Map()

  @volatile
  private var _valueInitialized: Boolean = false
  private var _value: Stream[KeyValuePair] = _

  @volatile
  private var _beanDefinitionInitialized: Boolean = false
  private var _beanDefinition: BeanDefinition = _

  protected def getValue(implicit ctx: EvaluationContext): Stream[KeyValuePair] = {
    if (!_valueInitialized) {
      synchronized {
        if (!_valueInitialized) {
          val kvp = getBeanDefinition.getAllProperties.toStream.map(property => {
            toJavaKeyValuePair(locationString, property)
          })
          _value = kvp
          _valueInitialized = true
        }
      }
    }
    _value
  }

  private def getBeanDefinition(implicit ctx: EvaluationContext): BeanDefinition = {
    if (!_beanDefinitionInitialized) {
      synchronized {
        if (!_beanDefinitionInitialized) {
          _beanDefinition = JavaBeanHelper.getBeanDefinition(bean)
          _beanDefinitionInitialized = true
        }
      }
    }
    _beanDefinition
  }

  private def toJavaKeyValuePair(loc: () => String, property: PropertyDefinition): KeyValuePair = {
    val name = property.name
    val javaValue: Value[_] = new JavaBeanPropertyValue(property, bean, JavaLocation.propertyAccess(loc, name))
    KeyValuePair(KeyValue(name), javaValue)
  }

  override def evaluate(implicit ctx: EvaluationContext): ObjectSeq = {
    new JavaBeanObjectSeq()
  }

  override def materialize(implicit ctx: EvaluationContext): Value[T] = {
    materializationRequested = true
    super.materialize
  }

  override def underlying()(implicit ctx: EvaluationContext): Any = bean

  override def underlyingClass(implicit ctx: EvaluationContext): Class[_] = clazz

  private class JavaBeanObjectSeq extends EagerObjectSeq with SimpleObjectSeq {

    override def apply(index: Long)(implicit ctx: EvaluationContext): KeyValuePair = {
      try {
        getValue.apply(index.toInt)
      } catch {
        case _: IndexOutOfBoundsException => null
      }
    }

    override def selectValue(key: Value[QualifiedName])(implicit ctx: EvaluationContext): Value[_] = {
      val name = key.evaluate.name
      if (cache.contains(name)) {
        cache.apply(name)
      } else {
        val property: PropertyDefinition = getBeanDefinition.getProperty(name)
        try {
          val value = property.read(bean, key)
          toWeaveValue(value, name)
        } catch {
          case _: InvalidPropertyNameException => null
        }
      }
    }

    override def selectKeyValue(key: Value[QualifiedName])(implicit ctx: EvaluationContext): KeyValuePair = {
      val name = key.evaluate.name
      if (cache.contains(name)) {
        KeyValuePair(KeyValue(name), cache.apply(name))
      } else {
        val property: PropertyDefinition = getBeanDefinition.getProperty(name)
        try {
          val value = property.read(bean, key)
          KeyValuePair(KeyValue(name), toWeaveValue(value, name))
        } catch {
          case _: InvalidPropertyNameException => null
        }
      }
    }

    private def toWeaveValue(value: Any, name: String)(implicit ctx: EvaluationContext): Value[_] = {
      val maybeValue = cache.get(name)
      if (maybeValue.isEmpty) {
        mapToWeaveValue(value, name)
      } else {
        maybeValue.get
      }
    }

    private def mapToWeaveValue(value: Any, name: String)(implicit ctx: EvaluationContext): Value[_] = {
      val javaValue = ReflectionJavaValueConverter.convert(value, () => name)
      // When materialization was requested and the value is BinaryValue,
      // the value must be materialized and cached to allow repeated access.
      if (materializationRequested && javaValue.isInstanceOf[BinaryValue]) {
        val binary = javaValue.asInstanceOf[BinaryValue]
        val materializedBinaryValue = binary.materialize
        cache.put(name, materializedBinaryValue)
        materializedBinaryValue
      } else {
        if (JavaValueHelper.isHeavyValue(javaValue)) {
          cache.put(name, javaValue)
        }
        javaValue
      }
    }

    override def allKeyValuesOf(key: Value[QualifiedName])(implicit ctx: EvaluationContext): Option[ObjectSeq] = {
      keyValueOf(key).map(kvp => ObjectSeq(kvp))
    }

    override def size()(implicit ctx: EvaluationContext): Long = {
      getValue.size
    }

    override def isEmpty()(implicit ctx: EvaluationContext): Boolean = {
      getValue.isEmpty
    }
  }
}

object JavaBeanObjectValue {
  def apply(bean: Any, loc: () => String): JavaBeanObjectValue = {
    new JavaBeanObjectValue(bean, loc)
  }
}
