package org.mule.weave.v2.module.pojo

import org.mule.weave.v2.model.service.SettingsService
import org.mule.weave.v2.module.java.reflection.AccessibilityObjectValidator
import org.mule.weave.v2.module.java.reflection.AlwaysCanAccessValidator
import org.mule.weave.v2.module.java.reflection.ClassHierarchyIterator
import org.mule.weave.v2.module.java.reflection.ValidateAccessibilityObjectValidator

import java.lang.reflect.Field
import java.lang.reflect.Method
import scala.annotation.tailrec
import scala.util.Try

trait BeanIntrospectionService {
  lazy val OBJECT_CLAZZ: Class[Object] = classOf[Object]

  def underlyingClass: Class[_]

  def traverseDeclaredMethodsInHierarchy(condition: Method => Boolean, callback: Method => Unit): Unit

  def findMethod(criteria: Method => Boolean): Option[Method]

  def getDeclaredFieldFromHierarchy(field: String): Option[Field]
}

class JpmsIntrospectionService(instance: Any, accessibilityObjectValidator: AccessibilityObjectValidator) extends BeanIntrospectionService {

  override def underlyingClass: Class[_] = instance.getClass

  override def traverseDeclaredMethodsInHierarchy(predicate: Method => Boolean, callback: Method => Unit): Unit = {
    val iterator = new ClassHierarchyIterator(underlyingClass)
    while (iterator.hasNext) {
      val clazz = iterator.next()
      if (!OBJECT_CLAZZ.equals(clazz)) {
        val methods = clazz.getDeclaredMethods
        methods.foreach(method => {
          if (predicate(method)) {
            // Only accessible method
            if (accessibilityObjectValidator.canAccess(method, instance)) {
              callback(method)
            }
          }
        })
      }
    }
  }

  override def findMethod(criteria: Method => Boolean): Option[Method] = {
    val it = new ClassHierarchyIterator(instance.getClass)
    var found: Method = null
    while (found == null && it.hasNext) {
      val clazz = it.next()
      val methods = clazz.getDeclaredMethods
      val maybeMethod = methods.find(criteria)
      if (maybeMethod.isDefined) {
        val method = maybeMethod.get
        // Only accessible method
        if (accessibilityObjectValidator.canAccess(method, instance)) {
          found = method
        }
      }
    }
    Option(found)
  }

  override def getDeclaredFieldFromHierarchy(field: String): Option[Field] = {
    val it = new ClassHierarchyIterator(instance.getClass)
    var found: Field = null
    while (found == null && it.hasNext) {
      val clazz = it.next()
      try {
        val maybeField = clazz.getDeclaredField(field)
        // Only accessible field
        if (accessibilityObjectValidator.canAccess(maybeField, instance)) {
          found = maybeField
        }
      } catch {
        case _: Exception =>
        // Try with next in the hierarchy
      }
    }
    Option(found)
  }
}

class ApiClassIntrospectionService(apiClazz: Class[_]) extends BeanIntrospectionService {

  override def underlyingClass: Class[_] = apiClazz

  override def traverseDeclaredMethodsInHierarchy(predicate: Method => Boolean, callback: Method => Unit): Unit = {
    @tailrec
    def doTraverse(apiClazz: Class[_], callback: Method => Unit): Unit = {
      val methods = apiClazz.getDeclaredMethods
      methods.foreach(method => {
        if (predicate(method)) {
          callback(method)
        }
      })
      val superclass = apiClazz.getSuperclass
      if (superclass != null && !OBJECT_CLAZZ.equals(superclass)) {
        doTraverse(superclass, callback)
      }
    }

    doTraverse(apiClazz, callback)
  }

  override def findMethod(criteria: Method => Boolean): Option[Method] = {
    apiClazz.getMethods.find(method => criteria(method))
  }

  override def getDeclaredFieldFromHierarchy(field: String): Option[Field] = {
    def innerGetDeclaredFieldFromHierarchy(name: String, apiClazz: Class[_]): Option[Field] = {
      if (apiClazz == null) {
        None
      } else {
        Try(
          Some(apiClazz.getDeclaredField(name))).getOrElse(innerGetDeclaredFieldFromHierarchy(name, apiClazz.getSuperclass))
      }
    }

    innerGetDeclaredFieldFromHierarchy(field, apiClazz)
  }
}

object BeanIntrospectionService {
  def apply(instance: Any, settingsService: SettingsService): BeanIntrospectionService = {
    if (settingsService.java().enabledClassDefinitionLookup) {
      val maybeApiClazz = ClassBeanDefinitionLookup.getClassDefinition(instance.getClass, settingsService)
      if (maybeApiClazz.isEmpty) {
        val accessor = if (settingsService.java().disableSetAccessible) {
          ValidateAccessibilityObjectValidator
        } else {
          AlwaysCanAccessValidator
        }
        createJpmsIntrospectionService(instance, accessor)
      } else {
        createApiClassIntrospectionService(maybeApiClazz.get)
      }
    } else {
      createApiClassIntrospectionService(instance.getClass)
    }
  }

  def createJpmsIntrospectionService(instance: Any, accessor: AccessibilityObjectValidator): BeanIntrospectionService = {
    new JpmsIntrospectionService(instance, accessor)
  }

  def createApiClassIntrospectionService(apiClazz: Class[_]): BeanIntrospectionService = {
    new ApiClassIntrospectionService(apiClazz)
  }
}

