package org.mule.weave.v2.module.pojo.reader

import org.mule.weave.v2.model.EvaluationContext
import org.mule.weave.v2.module.pojo.BeanIntrospectionService

import java.lang.reflect.Method
import java.lang.reflect.Modifier
import java.util.concurrent.ConcurrentHashMap
import java.util.concurrent.atomic.AtomicBoolean
import scala.collection.JavaConverters._

class BeanDefinition(beanIntrospectionService: BeanIntrospectionService) {
  private val requestedCache = new ConcurrentHashMap[String, PropertyDefinition](16, 0.9f, 1)
  private val readAccessorsCache = new ConcurrentHashMap[String, PropertyDefinition](16, 0.9f, 1)
  private val fullyLoaded: AtomicBoolean = new AtomicBoolean(false)

  def getProperty(name: String)(implicit ctx: EvaluationContext): PropertyDefinition = {
    // Load full read properties
    loadReadAccessors()

    val readAccessorPropertyName = NameGenerator.lowerJavaConvention(name)

    // This method potentially puts invalid properties into the cache, so we created a "requestedCache" to separate them
    // Invalid properties are possible because manually queried properties (with a selector) go through here
    // That is to allow fields without accessors
    var maybeDefinition = readAccessorsCache.get(readAccessorPropertyName)

    if (maybeDefinition == null) {
      maybeDefinition = requestedCache.get(name)
    }

    if (maybeDefinition == null) {
      val definition = PropertyDefinition(name, beanIntrospectionService, None, ctx.serviceManager.settingsService.java().honourBeanDefinitionAccessor)
      requestedCache.put(name, definition)
      definition
    } else {
      maybeDefinition
    }
  }

  def getAllProperties(implicit ctx: EvaluationContext): Iterable[PropertyDefinition] = {
    loadAllProperties()
    readAccessorsCache.values.asScala
  }

  private def loadAllProperties()(implicit ctx: EvaluationContext): Unit = {
    if (!fullyLoaded.get()) {
      loadReadAccessors()
      fullyLoaded.set(true)
    }
  }

  private final def loadReadAccessors()(implicit ctx: EvaluationContext): Unit = {
    beanIntrospectionService.traverseDeclaredMethodsInHierarchy(method => {
      // skip static methods.
      if (isValidModifier(method)) {
        val parameterCount = method.getParameterTypes.length
        val methodName = method.getName
        val accept = parameterCount match {
          case 0 if methodName.startsWith(BeanConstants.GET_PREFIX) && methodName.substring(BeanConstants.GET_PREFIX.length) != "" =>
            true
          case 0 if methodName.startsWith(BeanConstants.IS_PREFIX) =>
            true
          case _ =>
            false
        }
        accept
      } else {
        false
      }
    }, method => {
      val methodName = method.getName
      if (methodName.startsWith(BeanConstants.GET_PREFIX)) {
        cacheReadAccessors(NameGenerator.lowerJavaConvention(methodName.substring(BeanConstants.GET_PREFIX.length)), method)
      } else if (methodName.startsWith(BeanConstants.IS_PREFIX)) {
        cacheReadAccessors(NameGenerator.lowerJavaConvention(methodName.substring(BeanConstants.IS_PREFIX.length)), method)
      }
    })
  }

  private def isValidModifier(method: Method): Boolean = {
    val mods = method.getModifiers
    Modifier.isPublic(mods) && !Modifier.isNative(mods) && !Modifier.isStatic(mods)
  }

  private def cacheReadAccessors(name: String, readMethod: Method)(implicit ctx: EvaluationContext): Unit = {
    val maybeDefinition = readAccessorsCache.get(name)
    if (maybeDefinition == null) {
      val definition = PropertyDefinition(name, beanIntrospectionService, Some(readMethod), ctx.serviceManager.settingsService.java().honourBeanDefinitionAccessor)
      readAccessorsCache.put(name, definition)
    }
  }
}
