package org.mule.weave.v2.module.pojo.writer

import org.mule.weave.v2.core.exception.ExecutionException
import org.mule.weave.v2.model.EvaluationContext
import org.mule.weave.v2.model.structure.schema.Schema
import org.mule.weave.v2.model.types._
import org.mule.weave.v2.model.values._
import org.mule.weave.v2.model.values.wrappers.WrapperValue
import org.mule.weave.v2.module.DataFormat
import org.mule.weave.v2.module.commons.java.JavaClassLoaderHelper
import org.mule.weave.v2.module.commons.java.JavaTypesHelper.reg
import org.mule.weave.v2.module.commons.java.value.JavaSchema
import org.mule.weave.v2.module.commons.java.value.JavaValue
import org.mule.weave.v2.module.commons.java.writer.BaseJavaWriter
import org.mule.weave.v2.module.commons.java.writer.ClassTypeWithRestriction
import org.mule.weave.v2.module.commons.java.writer.JavaWriterSettings
import org.mule.weave.v2.module.commons.java.writer.converter.BaseJavaDataConverter
import org.mule.weave.v2.module.commons.java.writer.entry.JavaValueEntry
import org.mule.weave.v2.module.commons.java.writer.entry.ListEntry
import org.mule.weave.v2.module.commons.java.writer.entry.MapContainerEntry
import org.mule.weave.v2.module.commons.java.writer.entry.SimpleEntry
import org.mule.weave.v2.module.commons.java.writer.entry.WriterEntry
import org.mule.weave.v2.module.commons.java.writer.exception.CanNotConvertArrayException
import org.mule.weave.v2.module.java.ReflectionJavaClassLoaderHelper
import org.mule.weave.v2.module.pojo.JavaDataFormat
import org.mule.weave.v2.module.pojo.exception.CannotInstantiateException
import org.mule.weave.v2.module.pojo.exception.InterfaceWithNoBuilderException
import org.mule.weave.v2.module.pojo.writer.converter.JavaDataConverter
import org.mule.weave.v2.module.pojo.writer.entry.OptionalEntry
import org.mule.weave.v2.module.pojo.writer.entry._
import org.mule.weave.v2.module.pojo.writer.interceptor.JavaWriterInterceptor
import org.mule.weave.v2.module.reader.DefaultAutoPersistedOutputStream
import org.mule.weave.v2.parser.location.LocationCapable

import java.lang.reflect.Method
import java.util
import java.util.Optional
import java.util.concurrent.atomic.AtomicInteger
import scala.annotation.tailrec
import scala.util.Failure
import scala.util.Success
import scala.util.Try

class JavaWriter(override val loader: Option[ClassLoader], val settings: JavaWriterSettings) extends BaseJavaWriter {

  implicit val converter: BaseJavaDataConverter = JavaDataConverter

  override val classLoaderHelper: JavaClassLoaderHelper = ReflectionJavaClassLoaderHelper

  override protected def startArray(location: LocationCapable, schema: Option[Schema], clazzWithRestriction: ClassTypeWithRestriction)(implicit ctx: EvaluationContext): WriterEntry = {
    var arrayEntry: WriterEntry = null
    val clazz = clazzWithRestriction.classValue
    val contentRestriction = getParentGenericType(0, clazzWithRestriction, location)
    if (clazz.isDefined) {
      if (clazz.get.isArray) {
        arrayEntry = new ArrayEntry(location, clazz.get.getComponentType, schema)
      } else if (classOf[util.Collection[_]].isAssignableFrom(clazz.get) && canCreateInstance(clazz.get)) {
        arrayEntry = new ListEntry(location, clazz.get.newInstance().asInstanceOf[util.Collection[Any]], schema, contentRestriction)
      } else if (classOf[util.Iterator[Any]].isAssignableFrom(clazz.get)) {
        arrayEntry = new IteratorEntry(location, schema, contentRestriction)
      } else if (classOf[java.lang.Object].equals(clazz.get)) {
        arrayEntry = new ListEntry(location, new util.ArrayList[Any](), schema, contentRestriction)
      } else if (classOf[java.lang.Iterable[Any]].isAssignableFrom(clazz.get)) {
        arrayEntry = new IterableEntry(location, schema, contentRestriction)
      } else {
        throw new CanNotConvertArrayException(location.location(), clazz.get.getCanonicalName)
      }
    } else {
      arrayEntry = new ListEntry(location, new util.ArrayList[Any](), schema, contentRestriction)
    }
    arrayEntry
  }

  override protected def calculateClass(mayBeSchema: Option[Schema], locationCapable: LocationCapable)(implicit ctx: EvaluationContext): ClassTypeWithRestriction = {
    var result = super.calculateClass(mayBeSchema, locationCapable)

    if (result.classValue.isDefined && result.classValue.get.equals(classOf[Optional[_]])) {
      val contentType = getParentGenericType(0, result, locationCapable)
      entry.push(new OptionalEntry(locationCapable, contentType))
      result = contentType
    }

    result
  }

  private def endOptional(location: LocationCapable)(implicit ctx: EvaluationContext): Unit = {
    val pop: WriterEntry = entry.pop()
    write(pop)
  }

  override protected def endArray(location: LocationCapable)(implicit ctx: EvaluationContext): Unit = {
    val pop: WriterEntry = entry.pop()
    write(pop)
    if (entry.nonEmpty && entry.top.isInstanceOf[OptionalEntry]) endOptional(location)
  }

  @tailrec
  protected override final def doWriteValue(theValue: Value[_])(implicit ctx: EvaluationContext): Unit = {
    theValue match {
      case jv: JavaValue[_] if isAssignableToRequiredParentType(jv) => {
        //We check that is assignable to the expected entry type
        write(new JavaValueEntry(theValue, jv))
      }
      case selectedValue: WrapperValue => {
        doWriteValue(selectedValue.value)
      }
      case _ =>
        theValue.evaluate match {
          case it: ArrayType.T if theValue.schema.exists((s) => s.iterator.getOrElse(false)) => {
            write(new BeanContainerEntry(theValue, new JavaWriterIterator(loader.get, ctx, it.toIterator()), None))
          }
          case t: ArrayType.T => {
            val seq: Iterator[Value[_]] = t.toIterator()
            val schema = theValue.schema
            val mayBeClazz = calculateClass(schema, theValue)
            try {
              entry.push(startArray(theValue, schema, mayBeClazz))
              while (seq.hasNext) {
                val arrayItem = seq.next()
                writeValue(arrayItem)
              }
              endArray(theValue)
            } catch {
              case e: ExecutionException if (mayBeClazz.classValue.isDefined) => {
                val success = tryToWriteWithInterceptor(theValue, schema, mayBeClazz.classValue)
                if (!success) {
                  throw e
                }
              }
            }
          }
          case t: RangeType.T => {
            doWriteValue(ArrayType.coerce(theValue))
          }
          case t: ObjectType.T => {
            val schema = theValue.schema
            val mayBeClazz = calculateClass(schema, theValue)
            try {
              val writer = startObject(theValue, theValue.schema, mayBeClazz)
              entry.push(writer)
              val seq = t.toIterator()
              while (seq.hasNext) {
                val ekv = seq.next()
                key(ekv._1, ekv._1.evaluate, ekv._1.schema)
                writeAttributesAndValue(ekv, settings.writeAttributes)
                entry.pop()
              }
              endObject(theValue)
            } catch {
              case e: ExecutionException if (mayBeClazz.classValue.isDefined) => {
                val success = tryToWriteWithInterceptor(theValue, schema, mayBeClazz.classValue)
                if (!success) {
                  throw e
                }
              }
            }
          }
          case t: BinaryType.T => {
            val schemaOption = theValue.schema
            val binaryValue = BinaryType.coerce(theValue).evaluate
            val manager = ctx.serviceManager
            val target = new DefaultAutoPersistedOutputStream(manager.workingDirectoryService, manager.memoryService, manager.settingsService)
            binaryValue.copyTo(Some(target))
            writeSimpleJavaValue(target.toInputStream, theValue, schemaOption)
            if (entry.nonEmpty && entry.top.isInstanceOf[OptionalEntry]) endOptional(theValue)
          }
          case t if FunctionType.accepts(theValue) && fnWriterService(ctx).isDefined => {
            fnWriterService(ctx).get.write(theValue, this)
          }
          case t => {
            val schemaOption: Option[Schema] = theValue.schema
            writeSimpleJavaValue(t, theValue, schemaOption)
            if (entry.nonEmpty && entry.top.isInstanceOf[OptionalEntry]) endOptional(theValue)
          }
        }
    }
  }

  private def tryToWriteWithInterceptor(theValue: Value[_], schema: Option[Schema], mayBeClazz: Option[Class[_]])(implicit ctx: EvaluationContext) = {
    val maybeInterceptor = JavaWriterInterceptor.interceptorFor(mayBeClazz.get)
    val result = maybeInterceptor match {
      case Some(interceptor) => {
        val genericType = if (entry.isEmpty) None else Some(entry.top)
        write(new SimpleEntry(interceptor.write(theValue, genericType.map(_.genericType)), theValue, schema))
        true
      }
      case _ => false
    }
    result
  }

  def fnWriterService(ctx: EvaluationContext): Option[FunctionWriterService] = {
    ctx.serviceManager.lookupCustomService(classOf[FunctionWriterService])
  }

  override protected def startObject(location: Value[_], schema: Option[Schema], clazzWithRestriction: ClassTypeWithRestriction)(implicit ctx: EvaluationContext): WriterEntry = {
    var item: WriterEntry = null
    if (clazzWithRestriction.classValue.isDefined && isNotObjectClass(clazzWithRestriction.classValue.get)) {
      val clazz = clazzWithRestriction.classValue.get
      try {
        if (clazz.isInterface) {
          item = getBuilderMethod(clazz) match {
            case Failure(_) => {
              throw new InterfaceWithNoBuilderException(location.location(), clazz)
            }
            case Success(builderMethod) => {
              createBuilderEntry(clazz, builderMethod, location)
            }
          }
        } else {
          item = createWriterItem(location, schema, clazz, clazzWithRestriction)
        }
      } catch {
        case e: java.lang.InstantiationException => throw new CannotInstantiateException(location.location(), clazz.getName, Option(e.getMessage).getOrElse(""))
        case _: java.lang.IllegalAccessException => throw new CannotInstantiateException(location.location(), clazz.getName)
        case e: Exception => {
          throw new CannotInstantiateException(
            location.location(),
            clazz.getName,
            e.getClass.getName
              + " - " + Option(e.getMessage).getOrElse(""))
        }
      }
    } else {
      item = new MapContainerEntry(
        location,
        new java.util.LinkedHashMap[String, Any],
        schema,
        getParentGenericType(1, clazzWithRestriction, location).classValue.getOrElse(classOf[Object]),
        settings.duplicateKeyAsArray)
    }
    item
  }

  private def createBuilderEntry(clazz: Class[_], builderMethod: Method, location: Value[_])(implicit ctx: EvaluationContext): BuilderContainerEntry = {
    Try(builderMethod.invoke(null)) match {
      case Failure(_) => {
        throw new InterfaceWithNoBuilderException(location.location(), clazz)
      }
      case Success(builder) => {
        new BuilderContainerEntry(location, builder, clazz)
      }
    }
  }

  private def getBuilderMethod(clazz: Class[_]): Try[Method] = {
    Try(clazz.getMethod("builder"))
  }

  private def createWriterItem(location: Value[_], schema: Option[Schema], clazz: Class[_], clazzWithRestriction: ClassTypeWithRestriction)(implicit ctx: EvaluationContext): WriterEntry = {
    if (classOf[util.Map[_, _]].isAssignableFrom(clazz)) {
      new MapContainerEntry(
        location,
        clazz.newInstance().asInstanceOf[util.Map[String, Any]],
        schema,
        getParentGenericType(1, clazzWithRestriction, location).classValue.getOrElse(classOf[Object]),
        settings.duplicateKeyAsArray)
    } else {
      try {
        new BeanContainerEntry(location, clazz.newInstance(), schema)
      } catch {
        case ie: java.lang.InstantiationException => {
          getBuilderMethod(clazz) match {
            case Failure(_) => throw ie
            case Success(builderMethod) => {
              createBuilderEntry(clazz, builderMethod, location)
            }
          }
        }
      }
    }
  }

  private def writeSimpleJavaValue(valueToWrite: Any, location: LocationCapable, mayBeSchema: Option[Schema])(implicit ctx: EvaluationContext): Unit = {
    //If it has a class schema use it otherwise transform internal type into java ones
    mayBeSchema match {
      case Some(js: JavaSchema) => {
        val theValue = toJavaValue(valueToWrite, mayBeSchema, js.clazz, js.clazz.getSimpleName, location)
        write(new SimpleEntry(theValue, location, mayBeSchema))
      }
      case Some(schema) if schema.`class`.isDefined => {
        val currentClassName: String = schema.`class`.get
        val currentValueClassTree = classLoaderHelper.buildClassSchemaTree(reg.matcher(currentClassName).replaceAll(""), location)
        val currentClass = classLoaderHelper.loadClass(currentValueClassTree.className, loader, location)
        if (currentClass.equals(classOf[Optional[_]])) {
          val optionalValueType = getParentGenericType(0, ClassTypeWithRestriction(Some(currentClass), Some(currentValueClassTree)), location)
          entry.push(new OptionalEntry(location, optionalValueType))
          val theValue = toJavaValue(valueToWrite, mayBeSchema, optionalValueType.classValue.getOrElse(classOf[Object]), currentClassName, location)
          write(new SimpleEntry(theValue, location, mayBeSchema))
        } else {
          val theValue = toJavaValue(valueToWrite, mayBeSchema, currentClass, currentClassName, location)
          write(new SimpleEntry(theValue, location, mayBeSchema))
        }
      }
      case Some(schema) if NullType.acceptsValue(valueToWrite) && schema.inf.isDefined => {
        schema.inf match {
          case Some(NumberValue.NEGATIVE_INF_VALUE) => {
            write(new SimpleEntry(java.lang.Double.NEGATIVE_INFINITY, location, mayBeSchema))
          }
          case _ => {
            write(new SimpleEntry(java.lang.Double.POSITIVE_INFINITY, location, mayBeSchema))
          }
        }
      }
      case Some(schema) if NullType.acceptsValue(valueToWrite) && schema.nan.isDefined => {
        write(new SimpleEntry(java.lang.Double.NaN, location, mayBeSchema))
      }
      case _ => {
        write(new SimpleEntry(valueToWrite, location, mayBeSchema))
      }
    }
  }

  private def getParentGenericType(index: Int, contentTypeSchema: ClassTypeWithRestriction, location: LocationCapable)(implicit ctx: EvaluationContext): ClassTypeWithRestriction = {
    var clazz: Option[Class[_]] = if (contentTypeSchema.constrainClassSchema.isDefined && contentTypeSchema.constrainClassSchema.get.hasChild) {
      Some(classLoaderHelper.loadClass(contentTypeSchema.constrainClassSchema.get.firstChild.get.className, loader, location))
    } else {
      None
    }

    if (clazz.isEmpty && entry.nonEmpty) clazz = entry.top.genericType(index)

    ClassTypeWithRestriction(
      clazz,
      if (contentTypeSchema.constrainClassSchema.isDefined && contentTypeSchema.constrainClassSchema.get.hasChild)
        contentTypeSchema.constrainClassSchema.get.firstChild
      else
        None)
  }

  def canCreateInstance(entryType: Class[_]): Boolean = {
    !entryType.isInterface && isNotObjectClass(entryType) && !entryType.isSynthetic
  }

  def defaultConstructor(entryType: Class[_]): Boolean = {
    val value = entryType.getConstructor()
    value != null
  }

  def isNotObjectClass(entryType: Class[_]): Boolean = {
    entryType != classOf[Object]
  }

  override protected def endObject(location: LocationCapable)(implicit ctx: EvaluationContext): Unit = {
    val pop: WriterEntry = entry.pop()
    write(pop)
    if (entry.nonEmpty && entry.top.isInstanceOf[OptionalEntry]) endOptional(location)
  }

  override def dataFormat: Option[DataFormat[_, _]] = Some(JavaDataFormat)
}

object JavaWriter {
  def apply(loader: ClassLoader): JavaWriter = {
    val settings = JavaDataFormat.createWriterSettings()
    new JavaWriter(Some(loader), settings)
  }
}

/**
  * This is to adapt weave functions to Java ones
  * It counts the remaining functions that need to be executed before closing the context.
  *
  * It is a service because the count is shared mutable state that needs to reach all the written functions
  * so that the last one executed closes the context.
  *
  * Note: If a function is not closed, context won't be closed
  */
trait FunctionWriterService {
  val functionsBeforeClose: AtomicInteger = new AtomicInteger()

  def write(theValue: Value[_], writer: JavaWriter)(implicit ctx: EvaluationContext): Unit
}

