package org.mule.weave.v2.module.pojo.reader

import org.mule.weave.v2.model.EvaluationContext
import org.mule.weave.v2.module.pojo.exception.IllegalReadPropertyAccessException
import org.mule.weave.v2.module.pojo.exception.IllegalWritePropertyAccessException
import org.mule.weave.v2.module.pojo.exception.InvalidPropertyNameException
import org.mule.weave.v2.module.pojo.exception.JavaPropertyAccessException
import org.mule.weave.v2.module.pojo.exception.JavaPropertyWriteException
import org.mule.weave.v2.parser.location.LocationCapable

import java.lang.reflect.AccessibleObject
import java.lang.reflect.InvocationTargetException
import java.lang.reflect.Method
import java.lang.reflect.Type
import scala.util.Try

trait PropertyDefinition {

  private val baseName = NameGenerator.capitalizeJavaConvention(name)

  @volatile
  private var _readInitialized: Boolean = false
  private var _readMethod: Option[Method] = _

  @volatile
  private var _writeInitialized: Boolean = false
  private var _writeMethod: Option[Method] = _

  @volatile
  private var _classTypeInitialized: Boolean = false
  private var _classType: Class[_] = _

  def clazz: Class[_]

  def name: String

  def read(instance: Any, location: LocationCapable)(implicit ctx: EvaluationContext): Any = {
    try {
      doRead(instance, location)
    } catch {
      case e: InvalidPropertyNameException =>
        throw e
      case e @ (_: IllegalAccessException | _: IllegalArgumentException) =>
        val cause = e.getCause
        throw new IllegalReadPropertyAccessException(name, clazz.getName, location, if (cause != null) cause else e)
      case ite: InvocationTargetException =>
        throw new JavaPropertyAccessException(name, clazz.getName, location, ite.getTargetException)
      case e =>
        val cause = e.getCause
        throw new JavaPropertyAccessException(name, clazz.getName, location, if (cause != null) cause else e)
    }
  }

  protected def doRead(instance: Any, locationCapable: LocationCapable)(implicit ctx: EvaluationContext): Any

  def write(instance: Any, value: AnyRef, location: LocationCapable)(implicit ctx: EvaluationContext): Boolean = {
    try {
      doWrite(instance, value, location)
    } catch {
      case e @ (_: IllegalAccessException | _: IllegalArgumentException) =>
        val cause = e.getCause
        throw new IllegalWritePropertyAccessException(name, clazz.getName, location, if (cause != null) cause else e)
      case ite: InvocationTargetException =>
        throw new JavaPropertyWriteException(name, clazz.getName, location, ite.getTargetException)
      case e: Exception =>
        val cause = e.getCause
        throw new JavaPropertyWriteException(name, clazz.getName, location, if (cause != null) cause else e)
    }
  }

  protected def doWrite(instance: Any, value: AnyRef, location: LocationCapable)(implicit ctx: EvaluationContext): Boolean

  def genericType()(implicit ctx: EvaluationContext): Option[Type] = {
    readMethod.map(method => method.getGenericReturnType)
  }

  protected def readMethod(implicit ctx: EvaluationContext): Option[Method] = {
    if (!_readInitialized) {
      synchronized {
        if (!_readInitialized) {
          _readMethod = trySetAccessible(
            Option(
              Try(clazz.getMethod(BeanConstants.GET_PREFIX + baseName))
                .orElse(Try(clazz.getMethod(BeanConstants.IS_PREFIX + baseName)))
                .getOrElse(null)))
          _readInitialized = true
        }
      }
    }
    _readMethod
  }

  protected def writeMethod(implicit ctx: EvaluationContext): Option[Method] = {
    if (!_writeInitialized) {
      synchronized {
        if (!_writeInitialized) {
          val methodName = BeanConstants.SET_PREFIX + baseName
          val maybeMethod = clazz.getMethods.find(m => m.getName.equals(methodName) && m.getParameterTypes.length == 1)
          _writeMethod = trySetAccessible(maybeMethod)
          _writeInitialized = true
        }
      }
    }
    _writeMethod
  }

  protected def trySetAccessible[T <: AccessibleObject](accessibleObject: Option[T])(implicit ctx: EvaluationContext): Option[T] = {
    if (!ctx.serviceManager.settingsService.java().disableSetAccessible) {
      accessibleObject.foreach(m => {
        m.trySetAccessible()
      })
    }
    accessibleObject
  }

  def classType()(implicit ctx: EvaluationContext): Class[_] = {
    resolveClassType
  }

  private def resolveClassType(implicit ctx: EvaluationContext): Class[_] = {
    if (!_classTypeInitialized) {
      synchronized {
        if (!_classTypeInitialized) {
          _classType = doResolveClassType
          _classTypeInitialized = true
        }
      }
    }
    _classType
  }

  protected def doResolveClassType(implicit ctx: EvaluationContext): Class[_]
}

object PropertyDefinition {

  def apply(name: String, clazz: Class[_])(implicit ctx: EvaluationContext): PropertyDefinition = {
    if (ctx.serviceManager.settingsService.java().honourBeanDefinitionAccessor) {
      new DefaultPropertyDefinition(name, clazz)
    } else {
      new LegacyPropertyDefinition(name, clazz)
    }
  }
}
