package org.mule.weave.v2.module.pojo

import org.mule.weave.v2.model.EvaluationContext
import org.mule.weave.v2.module.DataFormat
import org.mule.weave.v2.module.commons.java.writer.JavaWriterSettings
import org.mule.weave.v2.module.commons.java.writer.ReadOnlyDefaultJavaWriterSettings
import org.mule.weave.v2.module.option.EmptySettings
import org.mule.weave.v2.module.option.Settings
import org.mule.weave.v2.module.pojo.reader.JavaReader
import org.mule.weave.v2.module.pojo.writer.JavaWriter
import org.mule.weave.v2.module.reader.SourceProvider
import org.mule.weave.v2.parser.module.MimeType

class JavaDataFormat extends DataFormat[Settings, JavaWriterSettings] {
  override def name(): String = "java"

  override def writer(target: Option[Any], outputMimeType: MimeType)(implicit ctx: EvaluationContext): JavaWriter = {
    target match {
      case Some(cl: ClassLoader) => new JavaWriter(Some(cl), createWriterSettings())
      case _                     => new JavaWriter(Some(Thread.currentThread().getContextClassLoader), createWriterSettings())
    }
  }

  /**
    * Creates a java writer configured with the default writer settings in a read only mode.
    * Use it if the writer settings are not going to be modified
    */
  def defaultSettingsWriter(target: Option[Any]): JavaWriter = {
    target match {
      case Some(cl: ClassLoader) => new JavaWriter(Some(cl), ReadOnlyDefaultJavaWriterSettings)
      case _                     => new JavaWriter(Some(Thread.currentThread().getContextClassLoader), ReadOnlyDefaultJavaWriterSettings)
    }
  }

  override def reader(source: SourceProvider)(implicit ctx: EvaluationContext): JavaReader = {
    new JavaReader(source.underling)
  }

  override val defaultMimeType: MimeType = {
    new MimeType("application", "java")
  }

  override val acceptedMimeTypes: Seq[MimeType] = {
    Seq(new MimeType("*", "java"))
  }

  /**
    * The file extensions associated with this DataFormat. (".json", ".xml", etc...)
    * Note: This can be empty
    */
  override def fileExtensions: Seq[String] = Seq()

  override def readerSettings(): Settings = EmptySettings

  override def writerSettings(): JavaWriterSettings = new JavaWriterSettings()
}

object JavaDataFormat extends JavaDataFormat {
  def isJavaMimeType(mimeTypeString: String): Boolean = {
    val mimeType = MimeType.fromSimpleString(mimeTypeString)
    JavaDataFormat.acceptedMimeTypes.exists((acceptedMimeType) => {
      acceptedMimeType.includes(mimeType)
    })
  }
}