package org.mule.weave.v2.module.javaplain.interpreted.node.structure.function

import org.mule.weave.v2.interpreted.ExecutionContext
import org.mule.weave.v2.interpreted.node.ValueNode
import org.mule.weave.v2.interpreted.node.structure.function.FunctionParameterNode
import org.mule.weave.v2.model.EvaluationContext
import org.mule.weave.v2.model.ServiceManager
import org.mule.weave.v2.model.capabilities.UnknownLocationCapable
import org.mule.weave.v2.model.values.Value
import org.mule.weave.v2.module.ConfigurableReaderWriter
import org.mule.weave.v2.module.javaplain.JavaPlainDataFormat
import org.mule.weave.v2.module.javaplain.api.contribution.JavaPlainBasedFunction
import org.mule.weave.v2.module.javaplain.api.contribution.ServiceProvider
import org.mule.weave.v2.module.javaplain.exception.JavaPlainBasedFunctionExecutionException
import org.mule.weave.v2.module.javaplain.writer.JavaPlainWriterSettings
import org.mule.weave.v2.module.reader.SourceProvider
import org.mule.weave.v2.module.writer.Writer
import org.mule.weave.v2.module.writer.WriterHelper
import org.mule.weave.v2.parser.ast.variables.NameIdentifier
import org.mule.weave.v2.parser.location.WeaveLocation
import org.mule.weave.v2.utils.Optionals.toJavaOptional

import java.io.PrintWriter
import java.io.StringWriter
import java.util.Optional
import scala.collection.mutable
import scala.language.postfixOps

class JavaPlainBasedFunctionBodyValueNode(functionFQNIdentifier: NameIdentifier, function: JavaPlainBasedFunction, parameters: Array[FunctionParameterNode], bodyLocation: WeaveLocation)
    extends ValueNode[Any] {

  private lazy val writerProperties = CaseInsensitivePropertiesTable(function.argument.writerProperties)

  private def extractFunctionArgumentValues()(implicit ctx: ExecutionContext): Array[Value[_]] = {
    val argsValues = new Array[Value[_]](parameters.length)
    var i = 0
    while (i < parameters.length) {
      val param = parameters(i)
      val arg = if (param.variable.module.isDefined) {
        val moduleName = param.variable.module.get
        ctx.executionStack().getVariable(moduleName.slot, param.variable.slot)
      } else {
        ctx.executionStack().getVariable(param.variable.slot)
      }
      argsValues.update(i, arg)
      i = i + 1
    }
    argsValues
  }

  private def createFunctionArgumentWriter()(implicit ctx: EvaluationContext): Writer = {
    val writer = JavaPlainDataFormat.writer(None)
    // Configure writer options
    configure(writer, writerProperties)

    // Configure passthrough option
    if (writer.settings.settingsOptions().containsOption(JavaPlainWriterSettings.PassthroughValueOptionName)) {
      writer.settings.set(JavaPlainWriterSettings.PassthroughValueOptionName, true)
    }
    writer
  }

  private def configure[T <: ConfigurableReaderWriter](configAware: T, properties: CaseInsensitivePropertiesTable)(implicit ctx: EvaluationContext): T = {
    // Configure the options based on the mimeType properties
    val options = configAware.settings.settingsOptions().definition.iterator
    while (options.hasNext) {
      val (name, moduleOption) = options.next()
      val maybeParam = properties.getProperty(name)
      if (maybeParam.isDefined) {
        configAware.setOption(location(), moduleOption.name, maybeParam.get)
      }
    }
    configAware
  }

  override protected def doExecute(implicit ctx: ExecutionContext): Value[Any] = {
    try {
      // Extract function argument from execution stack
      val functionArgs = extractFunctionArgumentValues()
      // Write every value argument before calling the function
      val args: Array[AnyRef] = functionArgs.map(arg => {
        val writer = createFunctionArgumentWriter()
        val (result, _) = WriterHelper.writeAndGetResult(writer, arg, UnknownLocationCapable)(ctx)
        result.asInstanceOf[AnyRef]
      })

      // Execute function call
      val result = function.call(args, new DefaultServicesProvider(ctx.serviceManager))

      // Read the function result
      val reader = JavaPlainDataFormat.reader(SourceProvider(result.value))
      val value = reader.read("JavaPlainBasedFunctionResult")
      value
    } catch {
      case e: Exception =>
        val stringWriter = new StringWriter()
        e.printStackTrace(new PrintWriter(stringWriter))
        throw new JavaPlainBasedFunctionExecutionException(functionFQNIdentifier, location(), s"${stringWriter.toString}")
    }
  }

  // This is for injected functions
  override def location(): WeaveLocation = bodyLocation
}

class DefaultServicesProvider(serviceManager: ServiceManager) extends ServiceProvider {
  override def lookupCustomService[T](service: Class[T]): Optional[T] = {
    serviceManager.lookupCustomService(service) asJava
  }
}

case class CaseInsensitivePropertiesTable(properties: java.util.Map[String, String]) {

  private lazy val props: Map[String, String] = {
    val map = new mutable.HashMap[String, String]
    properties.forEach((key, value) => {
      map.put(key.toLowerCase, value)
    })
    map.toMap
  }

  def getProperty(name: String): Option[String] = {
    props.get(name.toLowerCase)
  }
}
