package org.mule.weave.v2.module.javaplain.writer

import org.mule.weave.v2.core.io.SeekableStream
import org.mule.weave.v2.model.EvaluationContext
import org.mule.weave.v2.model.structure.schema.Schema
import org.mule.weave.v2.model.types.ArrayType
import org.mule.weave.v2.model.types.BinaryType
import org.mule.weave.v2.model.types.NullType
import org.mule.weave.v2.model.types.ObjectType
import org.mule.weave.v2.model.types.RangeType
import org.mule.weave.v2.model.values.NumberValue
import org.mule.weave.v2.model.values.Value
import org.mule.weave.v2.model.values.wrappers.WrapperValue
import org.mule.weave.v2.module.DataFormat
import org.mule.weave.v2.module.commons.java.JavaClassLoaderHelper
import org.mule.weave.v2.module.commons.java.JavaTypesHelper.reg
import org.mule.weave.v2.module.commons.java.value.JavaValue
import org.mule.weave.v2.module.commons.java.writer.BaseJavaWriter
import org.mule.weave.v2.module.commons.java.writer.ClassTypeWithRestriction
import org.mule.weave.v2.module.commons.java.writer.converter.BaseJavaDataConverter
import org.mule.weave.v2.module.commons.java.writer.entry.JavaValueEntry
import org.mule.weave.v2.module.commons.java.writer.entry.ListEntry
import org.mule.weave.v2.module.commons.java.writer.entry.MapContainerEntry
import org.mule.weave.v2.module.commons.java.writer.entry.SimpleEntry
import org.mule.weave.v2.module.commons.java.writer.entry.WriterEntry
import org.mule.weave.v2.module.commons.java.writer.exception.CanNotConvertArrayException
import org.mule.weave.v2.module.javaplain.JavaPlainDataFormat
import org.mule.weave.v2.module.reader.DefaultAutoPersistedOutputStream
import org.mule.weave.v2.parser.location.LocationCapable

import java.util

class JavaPlainWriter(val settings: JavaPlainWriterSettings) extends BaseJavaWriter {

  val loader: Option[ClassLoader] = None

  implicit val converter: BaseJavaDataConverter = BaseJavaDataConverter

  override implicit val classLoaderHelper: JavaClassLoaderHelper = JavaPlainClassLoaderHelper

  override protected def doWriteValue(theValue: Value[_])(implicit ctx: EvaluationContext): Unit = theValue match {
    case jv: JavaValue[_] if isAssignableToRequiredParentType(jv) =>
      write(new JavaValueEntry(jv, jv)) //We check that is assignable to the expected entry type
    case selectedValue: WrapperValue => doWriteValue(selectedValue.value)
    case _ => theValue.evaluate match {
      case t: ArrayType.T =>
        val seq: Iterator[Value[_]] = t.toIterator()
        val schema = theValue.schema
        val mayBeClazz = calculateClass(schema, theValue)
        entry.push(startArray(theValue, schema, mayBeClazz))
        while (seq.hasNext) {
          val arrayItem = seq.next()
          writeValue(arrayItem)
        }
        endArray(theValue)
      case _: RangeType.T => doWriteValue(ArrayType.coerce(theValue))
      case t: ObjectType.T =>
        val schema = theValue.schema
        val mayBeClazz = calculateClass(schema, theValue)
        val writer = startObject(theValue, theValue.schema, mayBeClazz)
        entry.push(writer)
        val seq = t.toIterator()
        while (seq.hasNext) {
          val ekv = seq.next()
          key(ekv._1, ekv._1.evaluate, ekv._1.schema)
          writeAttributesAndValue(ekv, settings.writeAttributes)
          entry.pop()
        }
        endObject(theValue)
      case _: BinaryType.T =>
        val schemaOption = theValue.schema
        val binaryValue = BinaryType.coerce(theValue).evaluate
        val seekableStream: SeekableStream = if (settings.passthroughValue) {
          binaryValue
        } else {
          val manager = ctx.serviceManager
          val target = new DefaultAutoPersistedOutputStream(manager.workingDirectoryService, manager.memoryService, manager.settingsService)
          binaryValue.copyTo(Some(target))
          target.toInputStream
        }
        writeSimpleJavaValue(seekableStream, theValue, schemaOption)
      case t =>
        val schemaOption: Option[Schema] = theValue.schema
        writeSimpleJavaValue(t, theValue, schemaOption)
    }
  }

  override protected def startArray(location: LocationCapable, schema: Option[Schema], clazzWithRestriction: ClassTypeWithRestriction)(implicit ctx: EvaluationContext): WriterEntry = {
    val clazz = clazzWithRestriction.classValue

    if (clazz.isDefined && !classOf[java.lang.Object].equals(clazz.get)) {
      throw new CanNotConvertArrayException(location.location(), clazz.get.getCanonicalName)
    }

    new ListEntry(location, new util.ArrayList[Any](), schema, clazzWithRestriction)
  }

  override protected def endArray(location: LocationCapable)(implicit ctx: EvaluationContext): Unit = write(entry.pop())

  override protected def startObject(location: Value[_], schema: Option[Schema], clazzWithRestriction: ClassTypeWithRestriction)(implicit ctx: EvaluationContext): WriterEntry = {
    new MapContainerEntry(
      location,
      new java.util.LinkedHashMap[String, Any],
      schema,
      classOf[Object],
      settings.duplicateKeyAsArray)
  }

  override protected def endObject(location: LocationCapable)(implicit ctx: EvaluationContext): Unit = write(entry.pop())

  private def writeSimpleJavaValue(valueToWrite: Any, location: LocationCapable, mayBeSchema: Option[Schema])(implicit ctx: EvaluationContext): Unit = mayBeSchema match {
    case Some(schema) if schema.`class`.isDefined =>
      val currentClassName: String = schema.`class`.get
      val currentValueClassTree = classLoaderHelper.buildClassSchemaTree(reg.matcher(currentClassName).replaceAll(""), location)
      val currentClass = classLoaderHelper.loadClass(currentValueClassTree.className, loader, location)
      val theValue = toJavaValue(valueToWrite, mayBeSchema, currentClass, currentClassName, location)
      write(new SimpleEntry(theValue, location, mayBeSchema))
    case Some(schema) if NullType.acceptsValue(valueToWrite) && schema.inf.isDefined =>
      schema.inf match {
        case Some(NumberValue.NEGATIVE_INF_VALUE) => {
          write(new SimpleEntry(java.lang.Double.NEGATIVE_INFINITY, location, mayBeSchema))
        }
        case _ => {
          write(new SimpleEntry(java.lang.Double.POSITIVE_INFINITY, location, mayBeSchema))
        }
      }
    case Some(schema) if NullType.acceptsValue(valueToWrite) && schema.nan.isDefined =>
      write(new SimpleEntry(java.lang.Double.NaN, location, mayBeSchema))
    case _ =>
      write(new SimpleEntry(valueToWrite, location, mayBeSchema))
  }

  override def dataFormat: Option[DataFormat[_, _]] = Some(new JavaPlainDataFormat)
}

object JavaPlainWriter {
  def apply(): JavaPlainWriter = {
    val settings = JavaPlainDataFormat.createWriterSettings()
    new JavaPlainWriter(settings)
  }
}
