package org.mule.weave.v2.module.javaplain.api.contribution

import org.mule.weave.v2.module.javaplain.api.contribution.NullaryJavaPlainBasedFunction.NULLARY_ARGUMENT
import org.mule.weave.v2.utils.Optionals.toJavaOptional

import java.util.Collections
import java.util.Optional
import scala.language.postfixOps

/**
  * Service instance with the [[JavaPlainBasedFunction]] contribution.
  */
trait JavaPlainBasedFunctionProviderService {
  /**
    * Returns the [[JavaPlainBasedFunction]] instance based on the given moduleFQNIdentifier and functionName.
    *
    * @param moduleFQNIdentifier The fully qualified name of the module identifier.
    * @param functionName The function local function name
    * @return The [[JavaPlainBasedFunction]] instance if exists.
    */
  def getFunction(moduleFQNIdentifier: String, functionName: String): Optional[JavaPlainBasedFunction]
}

/**
  * Representation of a Java Plain based function.
  */
trait JavaPlainBasedFunction {
  /**
    * @return The fully qualified name of the module identifier.
    */
  def moduleFQNIdentifier: String

  /**
    * @return The function local function name.
    */
  def functionName: String

  /**
    * @return The function argument.
    */
  def argument: JavaPlainBasedFunctionArgument

  /**
    * The function logic to be executed.
    * @param args The array of function arguments.
    * @param serviceProvider The service provider.
    * @return The function outcome.
    */
  def call(args: Array[AnyRef], serviceProvider: ServiceProvider): JavaPlainBasedFunctionResult
}

/**
  * Representation of a Java Plain based function without argument.
  */
trait NullaryJavaPlainBasedFunction extends JavaPlainBasedFunction {

  override final def argument: JavaPlainBasedFunctionArgument = NULLARY_ARGUMENT

  override final def call(args: Array[AnyRef], serviceProvider: ServiceProvider): JavaPlainBasedFunctionResult = {
    call(serviceProvider)
  }

  /**
    * The function logic to be executed.
    *
    * @param serviceProvider The service provider.
    * @return The function outcome.
    */
  def call(serviceProvider: ServiceProvider): JavaPlainBasedFunctionResult
}

object NullaryJavaPlainBasedFunction {
  private val NULLARY_ARGUMENT: JavaPlainBasedFunctionArgument = new JavaPlainBasedFunctionArgument {}
}

/**
  * Representation of a [[JavaPlainBasedFunction]] argument.
  */
trait JavaPlainBasedFunctionArgument {
  /**
    * @return The writer configuration properties used for write the function argument.
    */
  def writerProperties: java.util.Map[String, String] = Collections.emptyMap()
}

/**
  * Representation of a [[JavaPlainBasedFunction]] call.
  */
trait JavaPlainBasedFunctionResult {
  /**
    * @return The function outcome.
    */
  def value: AnyRef
}

/**
  * Provides access to the services available in the execution content.
  */
trait ServiceProvider {
  /**
    * Search for a custom service registered in the function execution context by the service [[Class]].
    *
    * @param service The [[Class]] service instance to look for.
    * @tparam T The service [[Class]] instance.
    * @return The custom service instance or `empty` if there is no custom service for the given service [[Class]].
    */
  def lookupCustomService[T](service: Class[T]): Optional[T]
}

/**
  * Builder of [[JavaPlainBasedFunctionProviderService]]s
  */
class JavaPlainBasedFunctionProviderServiceBuilder {

  private var functions = Map.empty[String, JavaPlainBasedFunction]

  /**
    * Includes a new [[JavaPlainBasedFunction]] instance.
    *
    * @param fn the [[JavaPlainBasedFunction]] to add.
    * @return this builder
    */
  def addFunction(fn: JavaPlainBasedFunction): JavaPlainBasedFunctionProviderServiceBuilder = {
    val key = functionKey(fn.moduleFQNIdentifier, fn.functionName)
    functions += (key -> fn)
    this
  }

  /**
    * @return Creates an instance of [[JavaPlainBasedFunctionProviderService]]
    */
  def build(): JavaPlainBasedFunctionProviderService = {
    new DefaultJavaPlainBasedFunctionProviderService(functions)
  }

  private def functionKey(module: String, function: String): String = {
    s"$module::$function"
  }

  private class DefaultJavaPlainBasedFunctionProviderService(functions: Map[String, JavaPlainBasedFunction]) extends JavaPlainBasedFunctionProviderService {

    override def getFunction(moduleFQNIdentifier: String, functionName: String): Optional[JavaPlainBasedFunction] = {
      val key = functionKey(moduleFQNIdentifier, functionName)
      functions.get(key) asJava
    }
  }
}

object JavaPlainBasedFunctionProviderServiceBuilder {
  def apply(): JavaPlainBasedFunctionProviderServiceBuilder = new JavaPlainBasedFunctionProviderServiceBuilder()
}
