package org.mule.weave.v2.module.javaplain.function

import org.mule.weave.v2.core.functions.TernaryFunctionValue
import org.mule.weave.v2.model.EvaluationContext
import org.mule.weave.v2.model.ServiceManager
import org.mule.weave.v2.model.capabilities.UnknownLocationCapable
import org.mule.weave.v2.model.types.ArrayType
import org.mule.weave.v2.model.types.StringType
import org.mule.weave.v2.model.values.Value
import org.mule.weave.v2.module.ConfigurableReaderWriter
import org.mule.weave.v2.module.javaplain.JavaPlainDataFormat
import org.mule.weave.v2.module.javaplain.api.contribution.JavaPlainBasedFunction
import org.mule.weave.v2.module.javaplain.api.contribution.JavaPlainBasedFunctionProviderService
import org.mule.weave.v2.module.javaplain.api.contribution.NullaryJavaPlainBasedFunction
import org.mule.weave.v2.module.javaplain.api.contribution.ServiceProvider
import org.mule.weave.v2.module.javaplain.contribution.EmptyJavaPlainBasedFunctionProviderService
import org.mule.weave.v2.module.javaplain.exception.InvalidJavaPlainBasedFunction
import org.mule.weave.v2.module.javaplain.exception.InvalidNullaryJavaPlainBasedFunctionArgumentException
import org.mule.weave.v2.module.javaplain.exception.JavaPlainBasedFunctionExecutionException
import org.mule.weave.v2.module.javaplain.writer.JavaPlainWriterSettings
import org.mule.weave.v2.module.reader.SourceProvider
import org.mule.weave.v2.module.writer.Writer
import org.mule.weave.v2.module.writer.WriterHelper
import org.mule.weave.v2.parser.ast.variables.NameIdentifier
import org.mule.weave.v2.utils.Optionals.toJavaOptional

import java.io.PrintWriter
import java.io.StringWriter
import java.util.Optional
import scala.collection.mutable
import scala.language.postfixOps

class JavaPlainInvokeFunction extends TernaryFunctionValue {

  override val First: StringType = StringType
  override val Second: StringType = StringType
  override val Third: ArrayType = ArrayType

  override protected def doExecute(firstValue: First.V, secondValue: Second.V, thirdValue: Third.V)(implicit ctx: EvaluationContext): Value[_] = {
    val moduleName = firstValue.evaluate.toString
    val functionName = secondValue.evaluate.toString
    val maybeJavaPlainBasedFunction = findJavaPlainBasedFunction(moduleName, functionName)
    if (maybeJavaPlainBasedFunction.isPresent) {
      val javaPlainBasedFunction = maybeJavaPlainBasedFunction.get
      val argsSeq = thirdValue.evaluate.toSeq()
      javaPlainBasedFunction match {
        case _: NullaryJavaPlainBasedFunction =>
          if (argsSeq.nonEmpty) {
            throw new InvalidNullaryJavaPlainBasedFunctionArgumentException(location())
          }
        case _ =>
        // Nothing to do!
      }
      doExecute(javaPlainBasedFunction, moduleName, functionName, argsSeq)
    } else {
      val functionFQNIdentifier = NameIdentifier(moduleName).child(functionName)
      throw new InvalidJavaPlainBasedFunction(functionFQNIdentifier, location())
    }
  }

  private def doExecute(function: JavaPlainBasedFunction, moduleName: String, functionName: String, functionArgs: Seq[Value[_]])(implicit ctx: EvaluationContext): Value[Any] = {
    try {
      // Write every value argument before calling the function
      val args: Array[AnyRef] = functionArgs.map(arg => {
        val writerProperties = CaseInsensitivePropertiesTable(function.argument.writerProperties)
        val writer = createFunctionArgumentWriter(writerProperties)
        val (result, _) = WriterHelper.writeAndGetResult(writer, arg, UnknownLocationCapable)(ctx)
        result.asInstanceOf[AnyRef]
      }).toArray

      // Execute function call
      val result = function.call(args, new DefaultServicesProvider(ctx.serviceManager))

      // Read the function result
      val reader = JavaPlainDataFormat.reader(SourceProvider(result.value))
      val value = reader.read("JavaPlainBasedFunctionResult")
      value
    } catch {
      case e: Exception =>
        val stringWriter = new StringWriter()
        e.printStackTrace(new PrintWriter(stringWriter))
        val functionFQNIdentifier = NameIdentifier(moduleName).child(functionName)
        throw new JavaPlainBasedFunctionExecutionException(functionFQNIdentifier, location(), s"${stringWriter.toString}")
    }
  }

  private def findJavaPlainBasedFunction(moduleFQNIdentifier: String, functionName: String)(implicit ctx: EvaluationContext): Optional[JavaPlainBasedFunction] = {
    val javaPlainBasedFunctionProviderService = ctx.serviceManager.lookupCustomService(classOf[JavaPlainBasedFunctionProviderService], EmptyJavaPlainBasedFunctionProviderService)
    javaPlainBasedFunctionProviderService.getFunction(moduleFQNIdentifier, functionName)
  }

  private def createFunctionArgumentWriter(writerProperties: CaseInsensitivePropertiesTable)(implicit ctx: EvaluationContext): Writer = {
    val writer = JavaPlainDataFormat.writer(None)
    // Configure writer options
    configure(writer, writerProperties)

    // Configure passthrough option
    if (writer.settings.settingsOptions().containsOption(JavaPlainWriterSettings.PassthroughValueOptionName)) {
      writer.settings.set(JavaPlainWriterSettings.PassthroughValueOptionName, true)
    }
    writer
  }

  private def configure[T <: ConfigurableReaderWriter](configAware: T, properties: CaseInsensitivePropertiesTable)(implicit ctx: EvaluationContext): T = {
    // Configure the options based on the mimeType properties
    val options = configAware.settings.settingsOptions().definition.iterator
    while (options.hasNext) {
      val (name, moduleOption) = options.next()
      val maybeParam = properties.getProperty(name)
      if (maybeParam.isDefined) {
        configAware.setOption(location(), moduleOption.name, maybeParam.get)
      }
    }
    configAware
  }
}

class DefaultServicesProvider(serviceManager: ServiceManager) extends ServiceProvider {
  override def lookupCustomService[T](service: Class[T]): Optional[T] = {
    serviceManager.lookupCustomService(service) asJava
  }
}

case class CaseInsensitivePropertiesTable(properties: java.util.Map[String, String]) {

  private lazy val props: Map[String, String] = {
    val map = new mutable.HashMap[String, String]
    properties.forEach((key, value) => {
      map.put(key.toLowerCase, value)
    })
    map.toMap
  }

  def getProperty(name: String): Option[String] = {
    props.get(name.toLowerCase)
  }
}