package org.mule.weave.v2.module.jsonschema

import org.everit.json.schema.ArraySchema
import org.everit.json.schema.BooleanSchema
import org.everit.json.schema.CombinedSchema
import org.everit.json.schema.ConditionalSchema
import org.everit.json.schema.ConstSchema
import org.everit.json.schema.EmptySchema
import org.everit.json.schema.EnumSchema
import org.everit.json.schema.FalseSchema
import org.everit.json.schema.NullSchema
import org.everit.json.schema.NumberSchema
import org.everit.json.schema.ObjectSchema
import org.everit.json.schema.ReferenceSchema
import org.everit.json.schema.Schema
import org.everit.json.schema.SchemaException
import org.everit.json.schema.StringSchema
import org.everit.json.schema.TrueSchema
import org.everit.json.schema.loader.SchemaClient
import org.everit.json.schema.loader.SchemaLoader
import org.json.JSONException
import org.json.JSONObject
import org.json.JSONTokener
import org.mule.weave.v2.core.RuntimeConfigProperties.TYPES_LOWER_CASE
import org.mule.weave.v2.core.versioning.SystemSetting
import org.mule.weave.v2.model.service.SystemPropertiesSettings
import org.mule.weave.v2.parser.MessageCollector
import org.mule.weave.v2.parser.SafeStringBasedParserInput
import org.mule.weave.v2.parser.annotation.EnclosedMarkAnnotation
import org.mule.weave.v2.parser.ast.header.directives.TypeDirective
import org.mule.weave.v2.parser.ast.header.directives.VersionDirective
import org.mule.weave.v2.parser.ast.header.directives.VersionMajor
import org.mule.weave.v2.parser.ast.header.directives.VersionMinor
import org.mule.weave.v2.parser.ast.module.ModuleNode
import org.mule.weave.v2.parser.ast.structure.BooleanNode
import org.mule.weave.v2.parser.ast.structure.NumberNode
import org.mule.weave.v2.parser.ast.structure.StringNode
import org.mule.weave.v2.parser.ast.structure.schema.SchemaNode
import org.mule.weave.v2.parser.ast.structure.schema.SchemaPropertyNode
import org.mule.weave.v2.parser.ast.types.IntersectionTypeNode
import org.mule.weave.v2.parser.ast.types.KeyTypeNode
import org.mule.weave.v2.parser.ast.types.KeyValueTypeNode
import org.mule.weave.v2.parser.ast.types.LiteralTypeNode
import org.mule.weave.v2.parser.ast.types.NameTypeNode
import org.mule.weave.v2.parser.ast.types.ObjectTypeNode
import org.mule.weave.v2.parser.ast.types.TypeReferenceNode
import org.mule.weave.v2.parser.ast.types.UnionTypeNode
import org.mule.weave.v2.parser.ast.types.WeaveTypeNode
import org.mule.weave.v2.parser.ast.variables.NameIdentifier
import org.mule.weave.v2.parser.location.UnknownLocation
import org.mule.weave.v2.parser.phase.ModuleLoader
import org.mule.weave.v2.parser.phase.ParsingContentInput
import org.mule.weave.v2.parser.phase.ParsingContext
import org.mule.weave.v2.parser.phase.ParsingResult
import org.mule.weave.v2.parser.phase.PhaseResult
import org.mule.weave.v2.sdk.NameIdentifierHelper
import org.mule.weave.v2.sdk.WeaveResourceResolver
import org.mule.weave.v2.sdk.WeaveResourceResolverAware
import org.mule.weave.v2.utils.WeaveNameHelper

import java.io.ByteArrayInputStream
import java.io.File
import java.io.FileInputStream
import java.io.InputStream
import java.lang
import java.net.URI
import java.net.URL
import java.util
import java.util.Map
import scala.collection.JavaConverters.mapAsScalaMapConverter
import scala.collection.convert.ImplicitConversions.`collection AsScalaIterable`
import scala.collection.convert.ImplicitConversions.`set asScala`
import scala.collection.mutable
import scala.collection.mutable.ArrayBuffer

class JsonSchemaModuleLoader extends ModuleLoader with WeaveResourceResolverAware {

  private val ROOT_TYPE_NAME = "Root"
  private var resolver: WeaveResourceResolver = _

  private def toWeaveTypeNode(schema: Schema, ctx: JsonSchemaTransformationContext): WeaveTypeNode = {
    schema match {
      case arr: ArraySchema => {
        val arrayItemType =
          if (arr.getItemSchemas != null && !arr.getItemSchemas.isEmpty) {
            val nodes = arr.getItemSchemas.toSeq.map((s) => toWeaveTypeNode(s, ctx))
            UnionTypeNode(nodes)
          } else {
            toWeaveTypeNode(arr.getAllItemSchema, ctx)
          }

        TypeReferenceNode(NameIdentifier("Array"), Some(Seq(arrayItemType)))
      }
      case _: StringSchema => {
        TypeReferenceNode(NameIdentifier("String"), None)
      }
      case _: NullSchema => {
        TypeReferenceNode(NameIdentifier("Null"), None)
      }
      case es: EnumSchema => {
        val values: util.Set[AnyRef] = es.getPossibleValues
        val elements: mutable.Set[WeaveTypeNode] = values.map((v) => {
          toLiteralType(v)
        })
        UnionTypeNode(elements.toSeq)
      }
      case _: TrueSchema => {
        LiteralTypeNode(BooleanNode(true.toString))
      }
      case _: FalseSchema => {
        LiteralTypeNode(BooleanNode(false.toString))
      }
      case _: BooleanSchema => {
        TypeReferenceNode(NameIdentifier("Boolean"), None)
      }
      case _: NumberSchema => {
        TypeReferenceNode(NameIdentifier("Number"), None)
      }
      case obt: ObjectSchema => {
        val propertySchemas: util.Map[String, Schema] = obt.getPropertySchemas
        val requiredProperties: util.List[String] = obt.getRequiredProperties
        val propertySet: util.Set[util.Map.Entry[String, Schema]] = propertySchemas.entrySet()
        val definedProperties: Array[KeyValueTypeNode] = propertySet.toArray(Array[util.Map.Entry[String, Schema]]()).map((p) => {
          val keyName = p.getKey
          val isRequired = requiredProperties.contains(keyName)
          KeyValueTypeNode(KeyTypeNode(NameTypeNode(Some(keyName))), toWeaveTypeNode(p.getValue, ctx), repeated = false, optional = !isRequired)
        })

        val additionPropertySchema: Seq[WeaveTypeNode] = if (obt.getSchemaOfAdditionalProperties != null) {
          Seq(toWeaveTypeNode(obt.getSchemaOfAdditionalProperties, ctx))
        } else {
          Seq()
        }
        val allAdditionalPropertiesTypes: Seq[WeaveTypeNode] = obt.getPatternProperties.asScala.toSeq
          .sortBy((p) => p._1.toString)
          .map((p) => {
            toWeaveTypeNode(p._2, ctx)
          }) ++ additionPropertySchema

        val additionalProperty: Seq[KeyValueTypeNode] =
          if (allAdditionalPropertiesTypes.isEmpty) {
            Seq()
          } else {
            val additionalPropertiesValueType: WeaveTypeNode = UnionTypeNode(allAdditionalPropertiesTypes)
            Seq(KeyValueTypeNode(KeyTypeNode(NameTypeNode(None)), additionalPropertiesValueType, repeated = false, optional = true))
          }
        val isOpen = obt.permitsAdditionalProperties()
        ObjectTypeNode(definedProperties ++ additionalProperty, None, None, !isOpen)
      }
      case ct: CombinedSchema if (ct.getCriterion() == CombinedSchema.ANY_CRITERION || ct.getCriterion() == CombinedSchema.ONE_CRITERION) => {
        val nodes = ct
          .getSubschemas()
          .toArray(Array[Schema]())
          .sortBy((s) => {
            s match {
              case _: ReferenceSchema => 5
              case cs: CombinedSchema if (cs.getCriterion == CombinedSchema.ALL_CRITERION) => 10
              case cs: CombinedSchema if (cs.getCriterion == CombinedSchema.ANY_CRITERION) => 20
              case cs: CombinedSchema if (cs.getCriterion == CombinedSchema.ONE_CRITERION) => 20
              case _: ConditionalSchema => 30
              case _ => 0
            }
          })
          .map((s) => toWeaveTypeNode(s, ctx))
        UnionTypeNode(nodes).annotate(EnclosedMarkAnnotation(UnknownLocation))
      }
      case ct: CombinedSchema if (ct.getCriterion() == CombinedSchema.ALL_CRITERION) => {
        val seq = ct
          .getSubschemas()
          .toArray(Array[Schema]())
          .sortBy((s) => {
            s match {
              case _: ReferenceSchema => 5
              case cs: CombinedSchema if (cs.getCriterion == CombinedSchema.ALL_CRITERION) => 10
              case cs: CombinedSchema if (cs.getCriterion == CombinedSchema.ANY_CRITERION) => 20
              case cs: CombinedSchema if (cs.getCriterion == CombinedSchema.ONE_CRITERION) => 20
              case _: ConditionalSchema => 30
              case _ => 0
            }
          })
        if (seq.length == 2) {
          if ((isSimpleType(seq(0)) && (seq(1).isInstanceOf[EnumSchema] || seq(1).isInstanceOf[ConstSchema]))) {
            return toWeaveTypeNode(seq(1), ctx)
          } else if ((isSimpleType(seq(1)) && (seq(0).isInstanceOf[EnumSchema] || seq(0).isInstanceOf[ConstSchema]))) {
            return toWeaveTypeNode(seq(0), ctx)
          }
        }
        val nodes = seq.map((s) => toWeaveTypeNode(s, ctx))
        IntersectionTypeNode(nodes).annotate(EnclosedMarkAnnotation(UnknownLocation))
      }
      case rt: ReferenceSchema => {
        TypeReferenceNode(NameIdentifier(ctx.nameOf(rt)), None)
      }
      case conditionalSchema: ConditionalSchema => {
        val thenSchema = conditionalSchema.getThenSchema
        val elseSchema = conditionalSchema.getElseSchema
        if (thenSchema.isPresent && elseSchema.isPresent) {
          UnionTypeNode(Seq(toWeaveTypeNode(thenSchema.get(), ctx), toWeaveTypeNode(elseSchema.get(), ctx)))
            .annotate(EnclosedMarkAnnotation(UnknownLocation))
        } else if (thenSchema.isPresent) {
          toWeaveTypeNode(thenSchema.get(), ctx)
        } else if (elseSchema.isPresent) {
          toWeaveTypeNode(elseSchema.get(), ctx)
        } else {
          TypeReferenceNode(NameIdentifier("Any"), None)
        }
      }
      case constSchema: ConstSchema => {
        val value = constSchema.getPermittedValue
        toLiteralType(value)
      }
      case _ => {
        TypeReferenceNode(NameIdentifier("Any"), None)
      }
    }
  }

  private def toLiteralType(v: AnyRef) = {
    v match {
      case s: String       => LiteralTypeNode(StringNode(s).withQuotation('\"'))
      case n: Number       => LiteralTypeNode(NumberNode(n.toString))
      case b: lang.Boolean => LiteralTypeNode(BooleanNode(b.toString))
      case _               => LiteralTypeNode(StringNode(String.valueOf(v)).withQuotation('\"'))
    }
  }

  private def isSimpleType(head: Schema) = {
    (head.isInstanceOf[StringSchema] || head.isInstanceOf[NumberSchema] || head.isInstanceOf[BooleanSchema])
  }

  private def toSchemaNode(schemaProperties: ArrayBuffer[SchemaPropertyNode]) = {
    if (schemaProperties.isEmpty) {
      None
    } else {
      Some(SchemaNode(schemaProperties))
    }
  }

  private def generateTypeDirectives(schema: Schema): Seq[TypeDirective] = {
    val context = new JsonSchemaTransformationContext()
    val value = toWeaveTypeNode(schema, context)
    val rootType = TypeDirective(NameIdentifier(ROOT_TYPE_NAME), None, value)
    val directives = new ArrayBuffer[TypeDirective]()
    directives += (rootType)
    while (context.hasMissingSchemas) {
      val schema = context.nextMissingSchema
      val newTypeNode = toWeaveTypeNode(schema.getReferredSchema, context)
      directives += TypeDirective(NameIdentifier(context.nameOf(schema)), None, newTypeNode)
    }
    directives
  }

  override def loadModule(nameIdentifier: NameIdentifier, parsingContext: ParsingContext): Option[PhaseResult[ParsingResult[ModuleNode]]] = {
    val path = getJsonPath(nameIdentifier)
    val maybeResource = resolver.resolvePath(path)
    maybeResource match {
      case Some(resource) => {
        val schemaData = resource.content()
        val baseURI = resource.url()
        try {
          val rawSchema: JSONObject = new JSONObject(new JSONTokener(schemaData))
          val schemaLoaderBuilder: SchemaLoader.SchemaLoaderBuilder = new SchemaLoader.SchemaLoaderBuilder()
          schemaLoaderBuilder.resolutionScope(baseURI)
          schemaLoaderBuilder.draftV7Support()
          schemaLoaderBuilder.schemaJson(rawSchema)
          schemaLoaderBuilder.schemaClient(new WeaveResolverSchemaClient(resolver))
          val schemaLoader: SchemaLoader = schemaLoaderBuilder.build
          val builder: Schema.Builder[_] = schemaLoader.load()
          val schema: Schema = builder.build().asInstanceOf[Schema]
          val typeDirectives: Seq[TypeDirective] = generateTypeDirectives(schema)
          val versionDirectives = Seq(VersionDirective(VersionMajor("2"), VersionMinor("0")))
          val moduleNode: ModuleNode = ModuleNode(nameIdentifier, versionDirectives ++ typeDirectives)
          val input: ParsingContentInput = ParsingContentInput(resource, nameIdentifier, SafeStringBasedParserInput(schemaData))
          Some(new PhaseResult(Some(ParsingResult(input, moduleNode)), MessageCollector()))
        } catch {
          case js: JSONException => {
            Some(new PhaseResult(None, MessageCollector().error(InvalidJsonSchemaMessage(js.getMessage), UnknownLocation)))
          }
          case se: SchemaException => {
            val message = se.getMessage
            val logMessage = if (message.startsWith(se.getSchemaLocation)) {
              message.substring(se.getSchemaLocation.length + 1).trim
            } else {
              message
            }
            Some(new PhaseResult(None, MessageCollector().error(InvalidJsonSchemaMessage(logMessage), UnknownLocation)))
          }
        }
      }
      case None => None
    }
  }

  private def getJsonPath(nameIdentifier: NameIdentifier): String = {
    NameIdentifierHelper.toWeaveFilePath(nameIdentifier, NameIdentifierHelper.fileSeparator, ".json")
  }

  override def name(): Option[String] = Some("jsonschema")

  override def resolver(resolver: WeaveResourceResolver): Unit = {
    this.resolver = resolver
  }

  private class WeaveResolverSchemaClient(weaveResourceResolver: WeaveResourceResolver) extends SchemaClient {

    override def get(url: String): InputStream = {
      val fileUrl = new URI(url)
      val maybeStream = weaveResourceResolver.resolve(NameIdentifierHelper.fromWeaveFilePath(fileUrl.getPath)) //
        .map(resource => new ByteArrayInputStream(resource.content().getBytes))
      maybeStream match {
        case Some(value) => value
        case None => if ("file".equals(fileUrl.getScheme) && new File(fileUrl).exists()) {
          new FileInputStream(new File(fileUrl))
        } else {
          throw new JSONException("Could not resolve referenced schema with URL: " + url)
        }
      }
    }
  }

  private class JsonSchemaTransformationContext() {
    private val AnonymousTypePrefix = "AnonymousType_"
    private val names: util.IdentityHashMap[Schema, String] = new util.IdentityHashMap[Schema, String]()
    private val missingSchemas = ArrayBuffer[ReferenceSchema]()

    def nameOf(ref: ReferenceSchema): String = {
      if (!names.containsKey(ref.getReferredSchema)) {
        val typeName = newName(ref)
        names.put(ref.getReferredSchema, typeName)
        missingSchemas.+=(ref)
        typeName
      } else {
        names.get(ref.getReferredSchema)
      }
    }

    private def buildAnonymousName = {
      var i = 0
      //This is because Identity hashMap uses identity equality
      while (names.values().toArray.contains(AnonymousTypePrefix + i)) {
        i = i + 1
      }
      AnonymousTypePrefix + i
    }

    private def newName(ref: ReferenceSchema) = {
      val value: String = ref.getReferenceValue
      value match {
        case v: String => {
          if (v.contains("#/")) { // #/ represents a local document reference
            val newName = WeaveNameHelper.toValidNameIdentifier(v.split("/").last, !TYPES_LOWER_CASE.get())
            if (names.values().contains(newName)) {
              buildAnonymousName
            } else {
              newName
            }
          } else {
            buildAnonymousName
          }
        }
        case _ => buildAnonymousName
      }

    }

    def hasMissingSchemas: Boolean = missingSchemas.nonEmpty

    def nextMissingSchema: ReferenceSchema = missingSchemas.remove(0)
  }

  override def canResolveModule(nameIdentifier: NameIdentifier): Boolean = resolver.canResolveResource(NameIdentifierHelper.fromWeaveFilePath(getJsonPath(nameIdentifier)))
}
