package org.mule.weave.lsp.project.impl.maven

import org.mule.weave.lsp.agent.WeaveAgentService
import org.mule.weave.lsp.jobs.JobManagerService
import org.mule.weave.lsp.project.Project
import org.mule.weave.lsp.project.ProjectKind
import org.mule.weave.lsp.project.ProjectKindContext
import org.mule.weave.lsp.project.ProjectKindDetector
import org.mule.weave.lsp.project.components.BuildManager
import org.mule.weave.lsp.project.components.MavenSampleDataManager
import org.mule.weave.lsp.project.components.MetadataProvider
import org.mule.weave.lsp.project.components.ModuleStructure
import org.mule.weave.lsp.project.components.ProjectDependencyManager
import org.mule.weave.lsp.project.components.ProjectStructure
import org.mule.weave.lsp.project.components.RootKind
import org.mule.weave.lsp.project.components.RootStructure
import org.mule.weave.lsp.project.components.SampleBaseMetadataProvider
import org.mule.weave.lsp.project.components.SampleDataManager
import org.mule.weave.lsp.project.components.TargetFolder
import org.mule.weave.lsp.project.components.TargetKind
import org.mule.weave.lsp.services.ClientLogger
import org.mule.weave.lsp.services.WeaveScenarioManagerService
import org.mule.weave.lsp.utils.EventBus
import org.mule.weave.lsp.utils.URLUtils.toURI
import org.mule.weave.lsp.utils.WeaveDirectoryUtils

import java.io.File

class MavenProjectKindDetector extends ProjectKindDetector {

  override def createKind(context: ProjectKindContext): ProjectKind = {
    new MavenProjectKind(
      context.project,
      new File(context.project.home(), "pom.xml"),
      context.eventBus,
      context.clientLogger,
      context.weaveAgentService,
      context.weaveScenarioManagerService,
      context.jobManagerService
    )
  }

  override def supports(context: ProjectKindContext): Boolean = {
    new File(context.project.home(), "pom.xml").exists()
  }
}

class MavenProjectKind(project: Project,
                       val pom: File,
                       eventBus: EventBus,
                       clientLogger: ClientLogger,
                       weaveAgentService: WeaveAgentService,
                       weaveScenarioManagerService: WeaveScenarioManagerService,
                       jobManagerService: JobManagerService) 
  extends ProjectKind {

  private lazy val projectDependencyManager = new MavenProjectDependencyManager(project, pom, eventBus, clientLogger, jobManagerService)
  private lazy val mavenBuildManager = new MavenBuildManager(pom, clientLogger, jobManagerService)
  private lazy val dataManager = new MavenSampleDataManager(this, project)
  private lazy val sampleBaseMetadataProvider = new SampleBaseMetadataProvider(weaveAgentService, eventBus, weaveScenarioManagerService)

  def this(pomFile: File, context: ProjectKindContext) = {
    this(
      context.project,
      pomFile,
      context.eventBus,
      context.clientLogger,
      context.weaveAgentService,
      context.weaveScenarioManagerService,
      context.jobManagerService
    )
  }

  override def name(): String = "DW `Maven`"

  override def structure(): ProjectStructure = {
    createProjectStructure()
  }

  override def isProjectFile(uri: String): Boolean = {
    toURI(uri).exists(uri => uri.normalize().compareTo(pom.toURI.normalize()) == 0)|| super.isProjectFile(uri)
  }

  private def createProjectStructure() = {
    val projectHome: File = project.home()
    val targetDir: File = new File(projectHome, "target")
    val rootStructures: Array[RootStructure] = Array(mainRoot(projectHome), testRoot(projectHome))
    val targets: Array[TargetFolder] = Array(
      TargetFolder(TargetKind.CLASS, Array(new File(targetDir, "classes"))),
      TargetFolder(TargetKind.TEST_CLASS, Array(new File(targetDir, "test-classes")))
    )
    ProjectStructure(Array(ModuleStructure(projectHome.getName, rootStructures, targets)))
  }

  protected def mainRoot(projectHome: File): RootStructure = {
    val mainDir = new File(projectHome, "src" + File.separator + "main")

    val mainSourceFolder = new File(mainDir, "dw")
    val srcFolders = Array(mainSourceFolder, new File(mainDir, "java")).filter(_.exists())
    val resourceFolders = Array(new File(mainDir, "resources")).filter(_.exists())

    val rootStructure = RootStructure(RootKind.MAIN, srcFolders, resourceFolders, mainSourceFolder)
    rootStructure
  }

  private def testRoot(projectHome: File): RootStructure = {
    val testDir = new File(projectHome, "src" + File.separator + "test")
    val mainSourceFolder = new File(testDir, WeaveDirectoryUtils.DWTest_FOLDER)
    val srcFolders = Array(mainSourceFolder).filter(_.exists())
    val resourceFolders = Array(new File(testDir, "resources")).filter(_.exists())
    val rootStructure = RootStructure(RootKind.TEST, srcFolders, resourceFolders, mainSourceFolder)
    rootStructure
  }

  override def dependencyManager(): ProjectDependencyManager = {
    projectDependencyManager
  }

  override def buildManager(): BuildManager = {
    mavenBuildManager
  }

  override def sampleDataManager(): SampleDataManager = {
    dataManager
  }

  override def metadataProvider(): Option[MetadataProvider] = {
    if (project.isStarted) {
      Some(sampleBaseMetadataProvider)
    } else {
      None
    }
  }
}
