package org.mule.weave.lsp.vfs

import org.apache.commons.io.FilenameUtils
import org.mule.weave.lsp.project.components.ProjectStructure
import org.mule.weave.lsp.utils.URLUtils
import org.mule.weave.v2.parser.ast.variables.NameIdentifier
import org.mule.weave.v2.sdk.NameIdentifierHelper

import java.io.File
import java.nio.file.Path

trait NameIdentifierResolver {
  def resolveNameIdentifier(url: String): Option[NameIdentifier]
}

class CompositeNameIdentifierResolver(resolvers: Array[NameIdentifierResolver]) extends NameIdentifierResolver {
  override def resolveNameIdentifier(url: String): Option[NameIdentifier] = {
    var i = 0
    var result: Option[NameIdentifier] = None
    while (result.isEmpty && i < resolvers.length) {
      result = resolvers(i).resolveNameIdentifier(url)
      i = i + 1
    }
    result
  }
}

object SimpleNameIdentifierResolver extends NameIdentifierResolver {
  override def resolveNameIdentifier(url: String): Option[NameIdentifier] = {
    val identifier = URLUtils.toFile(url)
      .map((f) => NameIdentifier(FilenameUtils.getBaseName(f.getName)))
      .getOrElse(NameIdentifierHelper.fromWeaveFilePath(url))
    Some(identifier) //This or
  }
}

class ProjectFileNameIdentifierResolver(projectStructure: ProjectStructure) extends NameIdentifierResolver {
  override def resolveNameIdentifier(url: String): Option[NameIdentifier] = {
    val rootFolders: Array[File] = projectStructure.modules
      .flatMap((module) => {
        module.roots.flatMap(_.sources) ++ module.roots.flatMap(_.resources)
      })
    val maybeFile: Option[File] = rootFolders
      .find((root) => {
        val maybePath: Option[Path] = URLUtils.toPath(url)
        maybePath
          .exists((path) => {
            path.startsWith(root.toPath)
          })
      })
    if (maybeFile.isDefined) {
      val relativePath = maybeFile.get.toPath.relativize(URLUtils.toPath(url).get)
      Some(NameIdentifierHelper.fromWeaveFilePath(relativePath.toString))
    } else {
      None
    }
  }
}

case class JarFileNameIdentifierResolver(jarScheme: String) extends NameIdentifierResolver {
  val ENTRY_SEPARATOR = "!"

  override def resolveNameIdentifier(url: String): Option[NameIdentifier] = {
    val maybeUri = URLUtils.toURI(url)
    maybeUri match {
      case Some(uri) if (uri.getScheme.equals(jarScheme) && uri.getPath.contains(ENTRY_SEPARATOR)) => {
        val parts = uri.getPath.split(ENTRY_SEPARATOR)
        Some(NameIdentifierHelper.fromWeaveFilePath(parts(1), "/"))
      }
      case _ => None
    }

  }
}

