package org.mule.weave.lsp

import com.google.gson.JsonObject
import org.eclipse.lsp4j.{CodeLensOptions, CompletionOptions, ExecuteCommandOptions, InitializeParams, InitializeResult, InitializedParams, ServerCapabilities, SignatureHelpOptions, TextDocumentSyncKind}
import org.eclipse.lsp4j.jsonrpc.services.{JsonDelegate, JsonNotification}
import org.eclipse.lsp4j.services.{LanguageClient, LanguageServer, WorkspaceService}
import org.mule.weave.lsp.agent.{SettingsBasedAgentClasspathResolver, WeaveAgentService}
import org.mule.weave.lsp.commands.CommandProvider
import org.mule.weave.lsp.extension.client.{NoopWeaveLanguageClient, WeaveLanguageClient}
import org.mule.weave.lsp.extension.services.{DependencyManagerService, WeaveTextDocumentService}
import org.mule.weave.lsp.indexer.LSPWeaveIndexService
import org.mule.weave.lsp.jobs.{JobManagerService, Status}
import org.mule.weave.lsp.project.{Project, ProjectKind, ProjectKindDetectorManager}
import org.mule.weave.lsp.project.commands.ProjectProvider
import org.mule.weave.lsp.project.components.ProjectDependencyManager
import org.mule.weave.lsp.project.events.ProjectStartedEvent
import org.mule.weave.lsp.services._
import org.mule.weave.lsp.services.delegate.{DependencyManagerServiceDelegate, TextDocumentServiceDelegate, WorkspaceServiceDelegate}
import org.mule.weave.lsp.utils.EventBus
import org.mule.weave.lsp.vfs.{CustomLoaderVirtualFileSystem, JarFileNameIdentifierResolver, LibrariesVirtualFileSystem, ProjectVirtualFileSystem}
import org.mule.weave.lsp.vfs.resource.CustomLoaderResourceResolver
import org.mule.weave.v2.editor.{CompositeFileSystem, SpecificModuleResourceResolver, VirtualFileSystem, WeaveToolingService}

import java.util
import java.util.concurrent.{CompletableFuture, ExecutorService}
import java.util.logging.{Level, Logger}
import scala.collection.JavaConverters._
import scala.collection.mutable.ArrayBuffer

class WeaveLanguageServer(previewServiceFactory: PreviewServiceFactory) extends LanguageServer {
  private val JAVA = "java"
  private val JSONSCHEMA = "jsonschema"
  private val JSON_EXTENSION = "json"
  private val XSD_EXTENSION = "xsd"
  private val XMLSCHEMA = "xmlschema"
  private val logger: Logger = Logger.getLogger(getClass.getName)
  private val eventbus = new EventBus(IDEExecutors.eventsExecutor())
  private val executorService = IDEExecutors.defaultExecutor()
  private val workspaceService: WorkspaceServiceDelegate = new WorkspaceServiceDelegate()
  private val dependencyManagerService: DependencyManagerServiceDelegate = new DependencyManagerServiceDelegate()
  private var globalFVS: VirtualFileSystem = _
  private val textDocumentService: TextDocumentServiceDelegate = new TextDocumentServiceDelegate()
  private var client: WeaveLanguageClient = NoopWeaveLanguageClient
  var weaveAgentService: WeaveAgentService = _
  var scenarioService: WeaveScenarioManagerService = _
  var descriptorProviderService: AgentBasedDataFormatDescriptorProviderService = _
  private var clientLogger: ClientLogger = _
  private var indexService: LSPWeaveIndexService = _
  private var projectValue: Project = _
  private var jobManagerService: JobManagerService = _
  protected var projectKind: ProjectKind = _
  private var weaveTestManager: DataWeaveTestService = _
  private var javaResourceResolver: CustomLoaderResourceResolver = _
  private var jsonschemaResourceResolver: CustomLoaderResourceResolver = _
  private var xmlschemaResourceResolver: CustomLoaderResourceResolver = _
  private var commandProvider: CommandProvider = _
  private val services: ArrayBuffer[ToolingService] = ArrayBuffer()
  protected var kindDetectorManager: ProjectKindDetectorManager = new ProjectKindDetectorManager()
  private var previewService: PreviewService = _

  def this() = {
    this(new DefaultPreviewServiceFactory)
  }

  def createWeaveToolingService(): WeaveToolingService = {
    val toolingService = new WeaveToolingService(globalFVS, descriptorProviderService, Array(SpecificModuleResourceResolver(JAVA, javaResourceResolver), SpecificModuleResourceResolver(JSONSCHEMA, jsonschemaResourceResolver), SpecificModuleResourceResolver(XMLSCHEMA, xmlschemaResourceResolver)))
    toolingService.withSymbolsIndexService(indexService)
    toolingService
  }

  def createDataWeaveToolingService(project: Project,
                                    languageClient: LanguageClient,
                                    vfs: VirtualFileSystem,
                                    documentServiceFactory: () => WeaveToolingService,
                                    executor: ExecutorService): DataWeaveToolingService = {
    new DataWeaveToolingService(project, languageClient, vfs, documentServiceFactory, executor)
  }

  def createWeaveScenarioManagerService(dataWeaveToolingService: DataWeaveToolingService,
                                        weaveLanguageClient: WeaveLanguageClient,
                                        virtualFileSystem: VirtualFileSystem): WeaveScenarioManagerService = {
    new WeaveScenarioManagerService(dataWeaveToolingService, weaveLanguageClient, virtualFileSystem)
  }

  def eventBus(): EventBus = eventbus

  def project(): Project = projectValue

  def createDocumentService(dataWeaveToolingService: DataWeaveToolingService,
                            executorService: ExecutorService,
                            projectVFS: ProjectVirtualFileSystem,
                            scenarioService: WeaveScenarioManagerService,
                            globalFVS: VirtualFileSystem,
                            commandProvider: CommandProvider): DataWeaveDocumentService = {
    new DataWeaveDocumentService(dataWeaveToolingService, executorService, projectVFS, scenarioService, globalFVS, commandProvider)
  }

  /**
    * Which prefix to use for the commands of this server (if any)
    */
  val commandsPrefix: Option[String] = None

  val previewScheme: String = "preview"
  val jarScheme: String = "jar"
  def createPreviewService(weaveAgentService: WeaveAgentService, client: WeaveLanguageClient, projectValue: Project, dataWeaveToolingService: DataWeaveToolingService, jobManagerService: JobManagerService, previewScheme: String): PreviewService = {
    previewServiceFactory.getPreviewService(weaveAgentService, client,projectValue, dataWeaveToolingService, jobManagerService, previewScheme)
  }
  override def initialize(params: InitializeParams): CompletableFuture[InitializeResult] = {
    CompletableFuture.supplyAsync(() => {
      Runtime.getRuntime.addShutdownHook(new Thread(() => this.shutdown().get()))

      logger.log(Level.INFO, s"Initialize($params)")
      clientLogger = new ClientLogger(client)
      projectValue = Project.create(params, eventbus)
      //
      javaResourceResolver = new CustomLoaderResourceResolver(() => weaveAgentService, JAVA, JAVA,eventBus())
      jsonschemaResourceResolver = new CustomLoaderResourceResolver(() => weaveAgentService, JSONSCHEMA, JSON_EXTENSION, eventBus())
      xmlschemaResourceResolver = new CustomLoaderResourceResolver(() => weaveAgentService, XMLSCHEMA, XSD_EXTENSION, eventBus())

      //Create FileSystem
      val jarFileNameIdentifierResolver = JarFileNameIdentifierResolver(jarScheme)
      val librariesVFS: LibrariesVirtualFileSystem = new LibrariesVirtualFileSystem(clientLogger, jarFileNameIdentifierResolver)
      val projectVFS: ProjectVirtualFileSystem = new ProjectVirtualFileSystem(jarFileNameIdentifierResolver)
      val customJavaLoaderVirtualFileSystem = new CustomLoaderVirtualFileSystem(javaResourceResolver, eventbus)
      val customXmlSchemaLoaderVirtualFileSystem = new CustomLoaderVirtualFileSystem(xmlschemaResourceResolver, eventbus)
      val customJsonSchemaLoaderVirtualFileSystem = new CustomLoaderVirtualFileSystem(jsonschemaResourceResolver, eventbus)
      globalFVS = new CompositeFileSystem(new CompositeFileSystem(new CompositeFileSystem(new CompositeFileSystem(projectVFS, librariesVFS), customJavaLoaderVirtualFileSystem), customXmlSchemaLoaderVirtualFileSystem), customJsonSchemaLoaderVirtualFileSystem)

      //Create Services
      val dataWeaveToolingService = createDataWeaveToolingService(projectValue, client, globalFVS, createWeaveToolingService, executorService)
      //TODO should this be enabled for all project kinds??
      //Or we should make the project kind enable it
      scenarioService = createWeaveScenarioManagerService(dataWeaveToolingService, client, globalFVS)
      val maybeSettings = Option(params.getInitializationOptions)
        .map(_.asInstanceOf[JsonObject])

      val agentClasspathResolver = new SettingsBasedAgentClasspathResolver(maybeSettings)
      weaveAgentService = new WeaveAgentService(dataWeaveToolingService, IDEExecutors.defaultExecutor(), clientLogger, projectValue, scenarioService, agentClasspathResolver)
      descriptorProviderService = new AgentBasedDataFormatDescriptorProviderService(weaveAgentService)

      jobManagerService = new JobManagerService(executorService, client)

      previewService = createPreviewService(weaveAgentService, client, projectValue, dataWeaveToolingService, jobManagerService, previewScheme)

      indexService = new LSPWeaveIndexService(clientLogger, client, projectVFS, jobManagerService)

      // Init The LSP Services And wire the implementation
      weaveTestManager = new DataWeaveTestService(client, projectVFS, clientLogger, jobManagerService, weaveAgentService, dataWeaveToolingService)

      commandProvider = new CommandProvider(
        globalFVS,
        projectVFS,
        clientLogger,
        client,
        projectValue,
        jobManagerService,
        dataWeaveToolingService,
        weaveAgentService,
        scenarioService,
        previewService,
        weaveTestManager,
        descriptorProviderService,
        commandsPrefix)

      val documentServiceImpl = createDocumentService(dataWeaveToolingService, executorService, projectVFS, scenarioService, globalFVS, commandProvider)
      textDocumentService.delegate = documentServiceImpl

      // Create the project
      projectKind = kindDetectorManager.detectProjectKind(
        projectValue, eventbus, clientLogger, weaveAgentService, client, scenarioService, jobManagerService, dataWeaveToolingService, previewService, commandProvider
      )

      clientLogger.logDebug("[DataWeave] Detected Project: " + projectKind.name())
      clientLogger.logDebug("[DataWeave] Project: " + projectKind.name() + " initialized ok.")

      val workspaceServiceImpl =
        new DataWeaveWorkspaceService(
          projectValue,
          globalFVS,
          clientLogger,
          client,
          jobManagerService,
          indexService,
          commandProvider
        )

      workspaceService.delegate = workspaceServiceImpl

      val dependencyManagerImpl = new DataWeaveDependencyManagerService(client)
      dependencyManagerService.delegate = dependencyManagerImpl

      services
        .++=(Seq(
          dependencyManagerImpl,
          workspaceServiceImpl,
          documentServiceImpl,
          weaveAgentService,
          jobManagerService,
          descriptorProviderService,
          dataWeaveToolingService,
          previewService,
          scenarioService,
          indexService,
          projectVFS,
          librariesVFS,
          weaveTestManager,
          commandProvider
        ))

      //Init the Services
      services.foreach(service => {
        service.init(projectKind, eventbus)
      })

      val capabilities = new ServerCapabilities
      capabilities.setTextDocumentSync(TextDocumentSyncKind.Full)
      capabilities.setCompletionProvider(new CompletionOptions(true, java.util.Arrays.asList(".")))
      capabilities.setHoverProvider(true)
      capabilities.setDocumentSymbolProvider(true)
      capabilities.setDefinitionProvider(true)
      capabilities.setDocumentFormattingProvider(true)
      capabilities.setFoldingRangeProvider(true)
      capabilities.setCodeActionProvider(true)
      capabilities.setWorkspaceSymbolProvider(true)
      capabilities.setCodeLensProvider(new CodeLensOptions(true))
      capabilities.setRenameProvider(true)
      capabilities.setDocumentRangeFormattingProvider(true)
      capabilities.setReferencesProvider(true)
      capabilities.setSignatureHelpProvider(new SignatureHelpOptions(util.Arrays.asList("("), util.Arrays.asList(",")))
      capabilities.setExecuteCommandProvider(new ExecuteCommandOptions(commandProvider.commandIds().asJava))
      new InitializeResult(capabilities)
    })
  }

  override def initialized(params: InitializedParams): Unit = {
    //Start the project
    jobManagerService.schedule((_: Status) => {
      try {
        clientLogger.logDebug(s"[DataWeave] Project Kind: ${projectKind.name()} Starting.")
        projectKind.start()
        val dependencyManager: ProjectDependencyManager = projectKind.dependencyManager()
        clientLogger.logDebug("[DataWeave] Dependency Manager: " + dependencyManager.getClass + " Starting.")
        dependencyManager.start()
        clientLogger.logDebug("[DataWeave] Dependency Manager: " + dependencyManager.getClass + " Started.")
        clientLogger.logDebug("[DataWeave] Starting other services.")
        //Start the Services
        services.foreach(service => {
          service.start()
        })
        //Mark project as initialized
        projectValue.markStarted
        eventbus.fire(new ProjectStartedEvent(projectValue))
        clientLogger.logDebug("[DataWeave] ProjectStartedEvent fired")
      } catch {
        case e: Exception =>
          clientLogger.logError("Unable to Start project.", e)
      }
    }, s"Starting DataWeave Project", s"Starting Project of Kind: ${projectKind.name()}.")
  }

  override def shutdown(): CompletableFuture[AnyRef] = {
    CompletableFuture.supplyAsync(() => {
      logger.log(Level.INFO, "Stopping the services")
      services.foreach(_.stop())
      null
    })
  }

  override def exit(): Unit = {
    System.exit(0)
  }

  @JsonDelegate
  override def getTextDocumentService: WeaveTextDocumentService = {
    logger.log(Level.INFO, "getTextDocumentService")
    textDocumentService
  }

  override def getWorkspaceService: WorkspaceService = {
    logger.log(Level.INFO, "getWorkspaceService")
    workspaceService
  }

  @JsonDelegate
  def getDependencyManager: DependencyManagerService = {
    logger.log(Level.INFO, "getDependencyManager")
    dependencyManagerService
  }

  def getTestService: DataWeaveTestService = {
    logger.log(Level.INFO, "getTestService")
    weaveTestManager
  }

  def getCommandProvider: CommandProvider = {
    logger.log(Level.INFO, "getCommandProvider")
    commandProvider
  }

  def connect(client: WeaveLanguageClient): Unit = {
    logger.log(Level.INFO, "connect")
    this.client = client
  }

  def getClient: WeaveLanguageClient = {
    this.client
  }

  def setClient(client: WeaveLanguageClient): Unit = {
    this.client = client
  }

  @JsonNotification("weave/project/create")
  def createProject(): Unit = {
    CompletableFuture.supplyAsync(() => {
      val projectProvider = new ProjectProvider(client, project())
      projectProvider.newProject()
    })

  }

  /**
    * Returns the File System for this language server
    *
    * @return Returns the VFS
    */
  def fs(): VirtualFileSystem = {
    if (globalFVS == null) {
      throw new IllegalStateException("Language server not yet initialized")
    }
    globalFVS
  }

  def executor(): ExecutorService = {
    this.executorService
  }

  def getIndexService() : LSPWeaveIndexService = {
    this.indexService
  }

  def getPreviewService() : PreviewService = {
    this.previewService
  }
}
