package org.mule.weave.lsp.agent

import com.google.gson.JsonObject
import org.apache.commons.io.FilenameUtils

import java.io.File
import java.io.FileFilter
import java.io.FileOutputStream
import java.nio.file.Files
import java.nio.file.Path
import java.util.logging.{Level, Logger}

trait AgentClasspathResolver {
  def resolveClasspathJars(): Seq[String]
}

class SettingsBasedAgentClasspathResolver(val settings: Option[JsonObject]) extends AgentClasspathResolver {
  val AGENT_FOLDER_KEY = "agentFolder"

  private lazy val resolver: AgentClasspathResolver = {
    val maybeFolder: Option[File] =
      settings.flatMap(s => {
        Option(s.get(AGENT_FOLDER_KEY)).map(_.getAsString)
      })
        .map(filePath => new File(filePath))
      
    if (maybeFolder.isDefined) {
      new FolderBasedAgentClasspathResolver(maybeFolder.get)
    } else {
      new ResourceBasedAgentClasspathResolver()
    }
  }

  override def resolveClasspathJars(): Seq[String] = {
    resolver.resolveClasspathJars()
  }
}

class ResourceBasedAgentClasspathResolver extends AgentClasspathResolver {
  private val LOGGER: Logger = Logger.getLogger(getClass.getName)
  
  private val TMP_DIR: File = new File(System.getProperty("java.io.tmpdir"))
  private val AGENT_FOLDER = "agent-server-libs"
  private val DESTINATION_FOLDER = "data-weave-agent"
  private val RESOURCES = Seq(s"$AGENT_FOLDER/agent-server.zip", s"$AGENT_FOLDER/agent-api.zip")
  private val JAR_EXTENSION = ".jar"
  
  lazy val maybeResolver: Option[AgentClasspathResolver] = {
    val maybeLibsDirectory = createLibsDirectory()
    maybeLibsDirectory.map(dir => new FolderBasedAgentClasspathResolver(dir))
  }

  private def createLibsDirectory(): Option[File] = {
    val url = getClass.getClassLoader.getResource(AGENT_FOLDER)
    if (url != null) {
      if (!TMP_DIR.exists()) {
        throw new RuntimeException("The specified temporary " + TMP_DIR.getAbsolutePath + " directory does not exits. Please create the directory or provide a different one.")
      }
      // Create target directory
      val targetDirectory = Files.createTempDirectory(DESTINATION_FOLDER)
      // Copy resources
      RESOURCES.foreach(resource => {
        copyJar(resource, targetDirectory)
      })
      Some(targetDirectory.toFile)
    } else {
      None
    }
  }
  
  private def copyJar(resource: String, targetDirectory: Path): Unit = {
    val is = getClass.getClassLoader.getResourceAsStream(resource)
    if (is != null) {
      try {
        val baseName = FilenameUtils.getBaseName(resource)
        val destination = targetDirectory.resolve(baseName + JAR_EXTENSION).toFile
        destination.createNewFile()
        destination.deleteOnExit()
        val out = new FileOutputStream(destination)
        try {
          val buffer = new Array[Byte](1024)
          Stream.continually(is.read(buffer)).takeWhile(_ != -1).foreach(out.write(buffer, 0, _))
        } finally {
          out.close()
        }
      } finally {
        is.close()
      }
    }
  }

  override def resolveClasspathJars(): Seq[String] = {
    val jars = if (maybeResolver.isDefined) {
      maybeResolver.get.resolveClasspathJars()
    } else {
      Seq.empty
    }
    if (jars.isEmpty) {
      LOGGER.log(Level.SEVERE, s"Could not resolve agent classpath jars")
    }
    jars
  }
}

class FolderBasedAgentClasspathResolver(directory: File) extends AgentClasspathResolver {
  val JAR_EXTENSION = "jar"
  
  lazy val jars: Seq[String] = {
    if (!directory.exists() || !directory.isDirectory) {
      Seq.empty
    } else {
      directory.listFiles(new FileFilter {
        override def accept(pathname: File): Boolean = {
          val isJar = FilenameUtils.getExtension(pathname.getName) == JAR_EXTENSION
          pathname.isFile && pathname.canRead && !pathname.isHidden && isJar
        }
      }).map(f => f.getAbsolutePath)
        .toSeq
    }
  }
  
  override def resolveClasspathJars(): Seq[String] = {
    jars
  }
}