package org.mule.weave.lsp.agent

import org.jboss.shrinkwrap.resolver.api.maven.coordinate.MavenCoordinates
import org.mule.weave.lsp.extension.client.PreviewResult
import org.mule.weave.lsp.project.Project
import org.mule.weave.lsp.project.ProjectKind
import org.mule.weave.lsp.project.components.ClassPathConfig
import org.mule.weave.lsp.project.components.DependencyArtifact
import org.mule.weave.lsp.project.components.InputMetadata
import org.mule.weave.lsp.project.components.JavaWeaveLauncher.buildJavaProcessBaseArgs
import org.mule.weave.lsp.project.components.ProjectStructure.mainResourcesFolders
import org.mule.weave.lsp.project.components.ProjectStructure.mainSourceFolders
import org.mule.weave.lsp.project.components.ProjectStructure.mainTargetFolders
import org.mule.weave.lsp.project.components.ProjectStructure.testsSourceFolders
import org.mule.weave.lsp.project.components.Scenario
import org.mule.weave.lsp.project.components.WeaveTypeBind
import org.mule.weave.lsp.project.events.DependencyArtifactResolvedEvent
import org.mule.weave.lsp.project.events.OnDependencyArtifactResolved
import org.mule.weave.lsp.project.events.OnProjectStarted
import org.mule.weave.lsp.project.events.ProjectStartedEvent
import org.mule.weave.lsp.services.ClientLogger
import org.mule.weave.lsp.services.DataWeaveToolingService
import org.mule.weave.lsp.services.ToolingService
import org.mule.weave.lsp.services.WeaveScenarioManagerService
import org.mule.weave.lsp.services.events.AgentStartedEvent
import org.mule.weave.lsp.services.events.AgentStoppedEvent
import org.mule.weave.lsp.utils.EventBus
import org.mule.weave.lsp.utils.IOUtils
import org.mule.weave.lsp.utils.NetUtils
import org.mule.weave.lsp.utils.ProcessStreamLineConsumer
import org.mule.weave.v2.agent.api.event.DataFormatsDefinitionsEvent
import org.mule.weave.v2.agent.api.event.ImplicitInputTypesEvent
import org.mule.weave.v2.agent.api.event.InferWeaveTypeEvent
import org.mule.weave.v2.agent.api.event.ModuleResolvedEvent
import org.mule.weave.v2.agent.api.event.PreviewExecutedEvent
import org.mule.weave.v2.agent.api.event.PreviewExecutedFailedEvent
import org.mule.weave.v2.agent.api.event.PreviewExecutedSuccessfulEvent
import org.mule.weave.v2.agent.api.event.UnexpectedServerErrorEvent
import org.mule.weave.v2.agent.api.event.WeaveDataFormatDescriptor
import org.mule.weave.v2.agent.api.event.WeaveDataFormatProperty
import org.mule.weave.v2.agent.client.ConnectionRetriesListener
import org.mule.weave.v2.agent.client.DataFormatDefinitionListener
import org.mule.weave.v2.agent.client.ImplicitWeaveTypesListener
import org.mule.weave.v2.agent.client.ModuleLoadedListener
import org.mule.weave.v2.agent.client.PreviewExecutedListener
import org.mule.weave.v2.agent.client.WeaveAgentClient
import org.mule.weave.v2.agent.client.WeaveTypeInferListener
import org.mule.weave.v2.agent.client.tcp.TcpClientProtocol
import org.mule.weave.v2.completion.DataFormatDescriptor
import org.mule.weave.v2.completion.DataFormatProperty
import org.mule.weave.v2.parser.ast.variables.NameIdentifier
import org.mule.weave.v2.sdk.WeaveResource
import org.mule.weave.v2.ts.WeaveType

import java.io.IOException
import java.util
import java.util.concurrent.CompletableFuture
import java.util.concurrent.CountDownLatch
import java.util.concurrent.Executor
import java.util.concurrent.TimeUnit
import java.util.concurrent.locks.Lock
import java.util.concurrent.locks.ReentrantLock
import scala.collection.JavaConverters.asScalaBufferConverter

/**
  * This service manages the WeaveAgent. This agent allows to query and execute scripts on a running DataWeave Engine.
  *
  */
class WeaveAgentService(validationService: DataWeaveToolingService,
                        executor: Executor,
                        clientLogger: ClientLogger,
                        project: Project,
                        scenarioManagerService: WeaveScenarioManagerService,
                        classpathResolver: AgentClasspathResolver
                       ) extends ToolingService {

  val AGENT_SERVER_LAUNCHER_MAIN_CLASS = "org.mule.weave.v2.agent.server.AgentServerLauncher"
  
  val LIGHT_GREY = "rgb(113, 113, 113)"
  val RED = "rgb(139, 0, 0)"

  private var agentProcess: Process = _
  private var weaveAgentClient: WeaveAgentClient = _
  private var projectKind: ProjectKind = _

  private val startAgentLock: Lock = new ReentrantLock()
  private var eventBus: EventBus = _

  override def init(projectKind: ProjectKind, eventBus: EventBus): Unit = {
    this.eventBus = eventBus
    this.projectKind = projectKind
    eventBus.register(DependencyArtifactResolvedEvent.ARTIFACT_RESOLVED, new OnDependencyArtifactResolved {
      override def onArtifactsResolved(artifacts: Array[DependencyArtifact]): Unit = {
        restart()
      }
    })

    eventBus.register(ProjectStartedEvent.PROJECT_STARTED, new OnProjectStarted {
      override def onProjectStarted(project: Project): Unit = {
        startAgent()
      }
    })
  }

  def restart(): Unit = {
    stopAgent()
    startAgent()
  }


  def startAgent(): Unit = {
    if (startAgentLock.tryLock()) {
      clientLogger.logDebug("Locked startAgent")
      try {
        if (!isProcessAlive) {
          val port: Int = NetUtils.freePort()
          val jars = classpathResolver.resolveClasspathJars()
          if (jars.isEmpty) {
            clientLogger.logError(s"[data-weave-agent] Could not resolve agent jars")
          }
          val classpathCfg = ClassPathConfig(shouldIncludeTarget = false, jars = jars)
          //No need to include the target as it sent on each run
          val commandArgs: util.ArrayList[String] = buildJavaProcessBaseArgs(projectKind, classpathCfg, AGENT_SERVER_LAUNCHER_MAIN_CLASS)
          val builder = new ProcessBuilder()
          val args = new util.ArrayList[String]()
          args.addAll(commandArgs)
          args.add("-p")
          args.add(port.toString)
          builder.command(args)
          agentProcess = builder.start()
          clientLogger.logDebug(s"[data-weave-agent] Starting Agent: ${args.asScala.mkString(" ")}")
          val processStreamLineConsumer = new ProcessStreamLineConsumer {

            override def onStdErrorLineRead(line: String): Unit = {
              clientLogger.logError(s"[data-weave-agent] $line")
            }


            override def onStdOutLineRead(line: String): Unit = {
              clientLogger.logDebug(s"[data-weave-agent] $line")
            }


            override def onIOException(ioException: IOException): Unit = {
              clientLogger.logError("[data-weave-agent] Error on Agent", ioException)
            }
          }

          IOUtils.forwardProcessStream(agentProcess, executor, processStreamLineConsumer)
          val clientProtocol = new TcpClientProtocol("localhost", port)
          weaveAgentClient = new WeaveAgentClient(clientProtocol)
          weaveAgentClient.connect(50, 500, new ConnectionRetriesListener {
            override def startConnecting(): Unit = {
              clientLogger.logDebug(s"[data-weave-agent] DataWeave Agent start connecting at: $port")
            }

            override def connectedSuccessfully(): Unit = {
              clientLogger.logDebug(s"[data-weave-agent] DataWeave Agent connected at: $port")
            }

            override def failToConnect(reason: String): Unit = {
              clientLogger.logError(s"[data-weave-agent] Fail to connect to agent: $reason")
            }

            override def onRetry(count: Int, total: Int): Boolean = {
              if (!isProcessAlive) {
                clientLogger.logError(s"[data-weave-agent] Will not retry as process is no longer alive and exit code was: `$agentExitCode`.")
                false
              } else {
                clientLogger.logError(s"[data-weave-agent] Retrying to connect: $count/$total.")
                true
              }
            }
          })
          if (!weaveAgentClient.isConnected) {
            clientLogger.logError(s"[data-weave-agent] Unable to connect to Weave Agent")
          } else {
            clientLogger.logDebug(s"[data-weave-agent] Weave Agent Started at port: `$port`.")
            eventBus.fire(new AgentStartedEvent())
          }
        }
      } finally {
        startAgentLock.unlock()
      }
    }
  }

  def getAgentVersion: Option[String] = {
    val runtime = projectKind.dependencyManager().getDependencies(groupId = Some("org.mule.weave"), artifactId = Some("runtime")).headOption

    runtime.map(f => MavenCoordinates.createCoordinate(f.artifactId).getVersion)
  }

  private def agentExitCode: Int = {
    if (agentProcess == null) -1 else agentProcess.exitValue()
  }

  private def isProcessAlive = {
    agentProcess != null && agentProcess.isAlive
  }

  override def start(): Unit = {}


  def inferInputMetadataForScenario(scenario: Scenario): CompletableFuture[Option[InputMetadata]] = {
    CompletableFuture.supplyAsync(() => {
      if (checkConnected()) {
        val result = new FutureValue[Option[InputMetadata]]()
        weaveAgentClient.inferInputsWeaveType(scenario.inputsDirectory().getAbsolutePath, new ImplicitWeaveTypesListener() {
          override def onImplicitWeaveTypesCalculated(event: ImplicitInputTypesEvent): Unit = {
            val binds: Array[WeaveTypeBind] = event.types.flatMap(m => {
              validationService.loadType(m.wtypeString).map(wt => WeaveTypeBind(m.name, wt))
            })
            result.set(Some(InputMetadata(binds)))
          }

          override def onUnexpectedError(unexpectedServerErrorEvent: UnexpectedServerErrorEvent): Unit = {
            clientLogger.logError(s"[data-weave-agent] Unexpected error at 'inferInputsWeaveType' caused by: ${unexpectedServerErrorEvent.stacktrace}")
            result.set(None)
          }
        })
        result.get().flatten
      } else {
        None
      }
    }, executor)
  }

  def resolveModule(identifier: String, loader: String): CompletableFuture[Option[WeaveResource]] = {
    CompletableFuture.supplyAsync(() => {
      if (checkConnected()) {
        val result = new FutureValue[Option[WeaveResource]]()
        weaveAgentClient.resolveModule(identifier, loader, calculateLocalClasspath(), new ModuleLoadedListener() {
          override def onModuleLoaded(resultEvent: ModuleResolvedEvent): Unit = {
            val maybeUrl = if (resultEvent.url == null) None else resultEvent.url //url may be null in old agents as it was added
            if (resultEvent.content.isDefined) {
              val content = resultEvent.content.getOrElse("")
              val url = maybeUrl.getOrElse("")
              result.set(Some(WeaveResource(url, content)))
            } else {
              result.set(None)
            }
          }

          override def onUnexpectedError(unexpectedServerErrorEvent: UnexpectedServerErrorEvent): Unit = {
            clientLogger.logError(s"[data-weave-agent] Unexpected error at 'resolveModule' caused by: ${unexpectedServerErrorEvent.stacktrace}")
            result.set(None)
          }
        })
        result.get().flatten
      } else {
        None
      }
    }, executor)
  }

  def checkConnected(): Boolean = {
    if (isDisconnected) {
      clientLogger.logDebug("[data-weave-agent] Restarting Agent as is not initialized.")
      restart()
    }
    weaveAgentClient != null && weaveAgentClient.isConnected
  }

  private def isDisconnected = {
    weaveAgentClient == null || !weaveAgentClient.isConnected || !isProcessAlive
  }

  def inferOutputMetadataForScenario(scenario: Scenario): CompletableFuture[Option[WeaveType]] = {
    CompletableFuture.supplyAsync(() => {
      if (checkConnected()) {
        val result = new FutureValue[Option[WeaveType]]()
        val maybeExpected = scenario.expected()
        if (maybeExpected.isDefined) {
          weaveAgentClient.inferWeaveType(maybeExpected.get.getAbsolutePath, new WeaveTypeInferListener() {
            override def onWeaveTypeInfer(event: InferWeaveTypeEvent): Unit = {
              result.set(validationService.loadType(event.typeString))
            }

            override def onUnexpectedError(unexpectedServerErrorEvent: UnexpectedServerErrorEvent): Unit = {
              clientLogger.logError(s"[data-weave-agent] Unexpected error at 'inferOutputMetadataForScenario' caused by: ${unexpectedServerErrorEvent.stacktrace}")
              result.set(None)
            }
          })
          result.get().flatten
        } else {
          None
        }
      } else {
        None
      }
    }, executor)
  }


  def dataFormats(): Array[WeaveDataFormatDescriptor] = {
    if (checkConnected()) {
      val runResult = new FutureValue[Array[WeaveDataFormatDescriptor]]()
      weaveAgentClient.definedDataFormats(new DataFormatDefinitionListener() {
        override def onDataFormatDefinitionCalculated(dfde: DataFormatsDefinitionsEvent): Unit = {
          runResult.set(dfde.formats)
        }

        override def onUnexpectedError(unexpectedServerErrorEvent: UnexpectedServerErrorEvent): Unit = {
          clientLogger.logError(s"[data-weave-agent] Unexpected error at 'definedDataFormats' caused by: ${unexpectedServerErrorEvent.stacktrace}")
          runResult.set(Array())
        }
      })
      runResult.get().getOrElse(Array())
    } else {
      Array()
    }
  }

  def run(nameIdentifier: NameIdentifier, content: String, url: String): PreviewResult = {
    val maybeScenario: Option[Scenario] = scenarioManagerService.activeScenario(nameIdentifier)
    run(nameIdentifier, content, url, maybeScenario, Map(), None, None, withTestsClasspath = false)
  }

  def run(
           nameIdentifier: NameIdentifier,
           content: String,
           url: String,
           previewScenario: Option[Scenario],
           runtimeProperties: Map[String, String],
           outputMimeType: Option[String],
           maybeTimeout: Option[Long],
           withTestsClasspath: Boolean
         ): PreviewResult = {
    if (checkConnected()) {
      val timeout: Long = maybeTimeout.getOrElse(project.settings.previewTimeout.value().toLong)
      val runResult = new FutureValue[PreviewResult](timeout * 2)
      val classpath = calculateLocalClasspath(withTestsClasspath)
      val inputsPath: String =
        previewScenario
          .map(_.inputsDirectory().getAbsolutePath)
          .getOrElse("")

      val startTime: Long = System.currentTimeMillis()

      val weaveAgentClientListener = new PreviewExecutedListener() {
        override def onPreviewExecuted(result: PreviewExecutedEvent): Unit = {
          val endTime = System.currentTimeMillis()
          result match {
            case PreviewExecutedFailedEvent(message, messages) =>
              val logsArray: Array[String] = messages.map(m => m.timestamp + " : " + m.message).toArray
              runResult.set(
                PreviewResult(
                  errorMessage = message,
                  success = false,
                  logs = util.Arrays.asList(logsArray: _*),
                  uri = url,
                  timeTaken = endTime - startTime
                )
              )
            case PreviewExecutedSuccessfulEvent(result, mimeType, extension, encoding, messages) =>
              val logsArray = messages.map(m => m.timestamp + " : " + m.message).toArray
              runResult.set(
                PreviewResult(content = new String(result, encoding),
                  mimeType = mimeType,
                  success = true,
                  logs = util.Arrays.asList(logsArray: _*),
                  uri = url,
                  timeTaken = endTime - startTime,
                  fileExtension = extension
                )
              )
          }
        }

        override def onUnexpectedError(unexpectedServerErrorEvent: UnexpectedServerErrorEvent): Unit = {
          val endTime = System.currentTimeMillis()
          clientLogger.logError(s"[data-weave-agent] Unexpected error at 'runPreview' caused by: ${unexpectedServerErrorEvent.stacktrace}")
          val result = PreviewResult(
            errorMessage = "Unexpected error at 'runPreview'",
            success = false,
            logs = util.Arrays.asList(unexpectedServerErrorEvent.stacktrace),
            uri =  url,
            timeTaken = endTime - startTime
          )
          runResult.set(result)
        }
      }
      weaveAgentClient.runPreview(inputsPath, content, nameIdentifier.toString(), url, timeout, classpath,
        runtimeProperties, outputMimeType, weaveAgentClientListener)

      runResult.get()
        .getOrElse({
          PreviewResult(errorMessage = "Unable to Start DataWeave Agent to Run Preview.", success = false, logs = util.Collections.emptyList(), uri = url)
        })
    } else {
      PreviewResult(errorMessage = "Unable to Start DataWeave Agent to Run Preview.", success = false, logs = util.Collections.emptyList(), uri = url)
    }
  }

  /**
    * Calculates the local classpath of the project
    * This is just the source folders (for local refresh) and the target (for things like java classes)
    *
    * @return
    */
  private def calculateLocalClasspath(withTests: Boolean = false): Array[String] = {
    val sources: Array[String] = mainSourceFolders(projectKind.structure()).map(_.getAbsolutePath)
    val resources: Array[String] = mainResourcesFolders(projectKind.structure()).map(_.getAbsolutePath)
    val targets: Array[String] = mainTargetFolders(projectKind.structure()).map(_.getAbsolutePath)
    val tests: Array[String] = if (withTests) testsSourceFolders(projectKind.structure()).map(_.getAbsolutePath) else Array()
    //Put sources first so that it detects the content of the currently changed dw files that are not yet copied into target
    sources ++ resources ++ targets ++ tests
  }

  def definedDataFormats(): CompletableFuture[Array[DataFormatDescriptor]] = {
    CompletableFuture.supplyAsync(() => {
      if (checkConnected()) {
        val result = new FutureValue[Array[DataFormatDescriptor]]()
        weaveAgentClient.definedDataFormats(new DataFormatDefinitionListener() {
          override def onDataFormatDefinitionCalculated(event: DataFormatsDefinitionsEvent): Unit = {
            val formats = event.formats
            val descriptor = formats.map(weaveDataFormatDescriptor => {
              val mimeType = weaveDataFormatDescriptor.mimeType
              DataFormatDescriptor(mimeType, weaveDataFormatDescriptor.id, toDataFormatProp(weaveDataFormatDescriptor.writerProperties), toDataFormatProp(weaveDataFormatDescriptor.readerProperties))
            })
            result.set(descriptor)
          }

          override def onUnexpectedError(unexpectedServerErrorEvent: UnexpectedServerErrorEvent): Unit = {
            clientLogger.logError(s"[data-weave-agent] Unexpected error at 'definedDataFormats' caused by: ${unexpectedServerErrorEvent.stacktrace}")
            result.set(Array.empty)
          }
        })
        result.get().getOrElse(Array.empty)
      } else {
        Array.empty
      }
    }, executor)
  }

  private def toDataFormatProp(weaveDataFormatPropertySeq: Array[WeaveDataFormatProperty]): Array[DataFormatProperty] = {
    weaveDataFormatPropertySeq.map(property => {
      DataFormatProperty(property.name, property.description, property.wtype, property.values)
    })
  }

  override def stop(): Unit = {
    clientLogger.logDebug("[data-weave-agent] stop() hit")
    stopAgent() 
  }

  private def stopAgent(): Unit = {
    clientLogger.logDebug("[data-weave-agent] stopAgent() hit")
    if (weaveAgentClient != null) {
      weaveAgentClient.disconnect()
    }
    if (agentProcess != null) {
      agentProcess.destroyForcibly()
      agentProcess = null
    }
    eventBus.fire(new AgentStoppedEvent())
  }
}

class FutureValue[T](maxTime: Long = 30000) {
  @volatile
  var value: T = _
  val countDownLatch = new CountDownLatch(1)

  def get(): Option[T] = {
    if (value == null) {
      countDownLatch.await(maxTime, TimeUnit.MILLISECONDS)
    }
    Option(value)
  }

  def set(v: T): Unit = {
    value = v
    countDownLatch.countDown()
  }
}
