package org.mule.weave.lsp.commands

import org.eclipse.lsp4j.ExecuteCommandParams
import org.mule.weave.lsp.commands.Commands.argAsString
import org.mule.weave.lsp.extension.client.WeaveLanguageClient
import org.mule.weave.lsp.project.ProjectKind
import org.mule.weave.lsp.project.components.ProjectStructure
import org.mule.weave.lsp.services.DataWeaveToolingService
import org.mule.weave.v2.editor.VirtualFileSystem
import org.mule.weave.v2.utils.StringHelper.toStringTransformer

import java.io.File
import scala.io.Source

class CreateIntegrationMappingFileCommand(val project: ProjectKind, val weaveLanguageClient: WeaveLanguageClient, val weaveDocumentToolingService: DataWeaveToolingService, val virtualFileSystem: VirtualFileSystem) extends AbstractCreateFileCommand {


  val MAPPING_TEMPLATE: String = {
    val source = Source.fromInputStream(getClass.getClassLoader.getResourceAsStream("dw-template-mapping.dwl"), "UTF-8")
    try {
      source.mkString
    } finally {
      source.close()
    }
  }


  override def getInputLabel: String = "Name of the Integration Mapping"

  override def getDefaultName(params: ExecuteCommandParams): String = {
    if (params.getArguments != null && params.getArguments.size() >= 2) {
      val moduleFileUri = argAsString(params.getArguments, 1)
      val file = virtualFileSystem.file(moduleFileUri)
      file.getNameIdentifier.localName().name + "IntegrationMapping.dwl"
    } else {
      "MyIntegrationMapping.dwl"
    }
  }


  override def getDefaultFolder: Option[File] = {
    ProjectStructure.defaultTestSourceFolder(project.structure())
  }

  def replaceImportsAndFunctionsTemplate(uri: String): String = {
    val file = virtualFileSystem.file(uri)
    val service = weaveDocumentToolingService.openDocument(uri, withExpectedOutput = true)
    val functionDefinitions = service.availableFunctions()
    val functionsString = functionDefinitions.map(definition => {
      val functionName = definition.nameIdentifier.nameElements().last
      val params = definition.params.map(param => param.defaultValue.getOrElse("???")).mkString(",")
      s"""|    ${functionName} : ${functionName}(${params})"""
    }).mkString("," + System.lineSeparator())
    s"""|%dw ${project.getWeaveVersion()}
       |import * from ${file.getNameIdentifier.name}
       |
       |output application/json
       |---
       |{
       ${functionsString}
       |}"""
  }

  override def getTemplate(executeCommandParams: ExecuteCommandParams): String = {
    val template = if (executeCommandParams.getArguments != null && executeCommandParams.getArguments.size() >= 2) {
      val moduleUri = argAsString(executeCommandParams.getArguments, 1)
      replaceImportsAndFunctionsTemplate(moduleUri)
    } else {
      MAPPING_TEMPLATE.replace("${syntaxVersion}", project.getWeaveVersion())
    }
    s"""${INTEGRATION_MAPPING_CONTENT}
    ${template}""".stripMarginAndNormalizeEOL
  }

  /**
    * The Id of the underlying command
    *
    * @return
    */
  override def commandId(): String = Commands.DW_CREATE_TEST_MAPPING

  /**
    * Returns the name of the command
    */
  override def name(): String = "Create Integration Mapping"

  /**
    * Returns a description of the current command is doing. This message is going to be presented to the user as information of
    * what the command is running
    *
    * @return A short description of what this command is doing
    */
  override def description(params: ExecuteCommandParams): String = "Creating Integration Mapping File."
}
