package org.mule.weave.lsp.extension.client

import org.eclipse.lsp4j.jsonrpc.services.JsonNotification
import org.eclipse.lsp4j.jsonrpc.services.JsonRequest
import org.eclipse.lsp4j.services.LanguageClient

import java.util
import java.util.UUID
import java.util.concurrent.CompletableFuture
import javax.annotation.Nullable

trait WeaveLanguageClient extends LanguageClient {


  /**
    * Opens an input box to ask the user for input.
    *
    * @return the user provided input. The future can be cancelled, meaning
    *         the input box should be dismissed in the editor.
    */
  @JsonRequest("weave/inputBox")
  def weaveInputBox(params: WeaveInputBoxParams): CompletableFuture[WeaveInputBoxResult]

  /**
    * Opens an menu to ask the user to pick one of the suggested options.
    *
    * @return the user provided pick. The future can be cancelled, meaning
    *         the input box should be dismissed in the editor.
    */
  @JsonRequest("weave/quickPick")
  def weaveQuickPick(params: WeaveQuickPickParams): CompletableFuture[WeaveQuickPickResult]


  @JsonRequest("weave/fileBrowser")
  def weaveFileBrowser(params: WeaveOpenDialogParams): CompletableFuture[String]


  @JsonNotification("weave/decorations/set")
  def setEditorDecorations(params: EditorDecorationsParams): Unit

  @JsonNotification("weave/decorations/clear")
  def clearEditorDecorations(): Unit

  /**
    * Opens a folder in a new window
    *
    * @param params
    */
  @JsonNotification("weave/folder/open")
  def openWindow(params: OpenWindowsParams): Unit

  /**
    * This notification is sent from the server to the client to execute the specified configuration on client side.
    *
    */
  @JsonNotification("weave/workspace/run")
  def runConfiguration(config: LaunchConfiguration): Unit

  /**
    * This notification is sent from the server to the client to open an editor with the specified document uri
    *
    * @param params The document to be opened
    */
  @JsonNotification("weave/workspace/openTextDocument")
  def openTextDocument(params: OpenTextDocumentParams): Unit

  /**
    * This notification is sent from the server to the client to set context variable on visual studio code
    * used for enabling/disabling commands
    * at the editor with the specified document uri
    *
    * @param params The document to be opened
    */
  @JsonNotification("weave/workspace/setContext")
  def setContext(params: SetContextParams): Unit

  /**
    * This notification is sent from the server to the client to show the live data of a script
    *
    * @param result The result of executing a script
    */
  @JsonNotification("weave/workspace/showPreviewResult")
  def showPreviewResult(result: PreviewResult): Unit

  /**
    * This notification is sent from the server to the client to publish all the resolved dependencies of this workspace
    *
    * @param resolvedDependency The list of all the resolved dependencies
    */
  @JsonNotification("weave/workspace/publishDependencies")
  def publishDependencies(resolvedDependency: DependenciesParams): Unit

  /**
    * This notification is sent from the server to the client to publish current transformation scenarios.
    *
    * @param scenariosParam Scenarios Parameter
    */
  @JsonNotification("weave/workspace/publishScenarios")
  def showScenarios(scenariosParam: ShowScenariosParams): Unit

  /**
    * This notification is sent from the server to the client to inform the user that a background job has started.
    *
    * @param job The job information that has started
    */
  @JsonNotification("weave/workspace/notifyJobStarted")
  def notifyJobStarted(job: JobStartedParams): Unit

  /**
    * This notification is sent from the server to the client to inform the user that a background job has finish.
    *
    * @param job The job information that has ended
    */
  @JsonNotification("weave/workspace/notifyJobEnded")
  def notifyJobEnded(job: JobEndedParams): Unit


  /**
    * This notification is sent from the server to the client to push all the possible tests to run on the project.
    *
    * @param job The job information that has ended
    */
  @JsonNotification("weave/tests/publishTestItems")
  def publishTestItems(job: PublishTestItemsParams): Unit

  /**
    * This notification is sent from the server to the client to push tests results.
    *
    * @param job The job information that has ended
    */
  @JsonNotification("weave/tests/publishTestResults")
  def publishTestResults(testResults: PublishTestResultsParams): Unit
}

case class PublishTestResultsParams(event: String,message: String, name: String, duration: Int, locationHint: String, status: String)

case class PublishTestItemsParams(rootTestItems: java.util.List[WeaveTestItem])

case class WeaveTestItem(id: String = UUID.randomUUID().toString, label: String, uri: String, children: java.util.List[WeaveTestItem] = new util.ArrayList[WeaveTestItem](), @Nullable range: org.eclipse.lsp4j.Range = null)

case class JobStartedParams(id: String = UUID.randomUUID().toString, label: String, description: String)

case class JobEndedParams(id: String)

case class DependenciesParams(
                               //The list of dependencies
                               dependencies: util.List[DependencyDefinition]
                             )

case class DependencyDefinition(
                                 // The id of this dependency
                                 id: String,
                                 // The uri where this dependency can be located
                                 uri: String
                               )


case class SetContextParams(
                             contextValues: java.util.List[SetContextValue],
                                      )

case class SetContextValue(
                             contextKey: String,
                             enabled: Boolean,
                             contextFileUri: String
                           )
case class PreviewResult(
                          uri: String,
                          success: Boolean,
                          logs: java.util.List[String],
                          content: String = null,
                          mimeType: String = null,
                          errorMessage: String = null,
                          scenarioUri: String = null,
                          fileExtension: String = ".txt",
                          timeTaken: Long = 0
                        )


case class LaunchConfiguration(
                                //The type of config
                                `type`: String,
                                //The name of the config
                                name: String,
                                //launch, attach
                                request: String,
                                //If true it will no execute in debug mode
                                noDebug: Boolean,
                                //The additional properties used for this configuration
                                properties: java.util.List[LaunchConfigurationProperty])


object LaunchConfiguration {
  val DATA_WEAVE_CONFIG_TYPE_NAME = "data-weave"
  val WTF_CONFIG_TYPE_NAME = "data-weave-testing"
  val BAT_CONFIG_TYPE_NAME = "bat"
  val WITF_CONFIG_TYPE_NAME = "data-weave-integration-testing"

  val TYPE_PROP_NAME = "type"
  val REQUEST_PROP_NAME = "request"
  val NAME_PROP_NAME = "name"
  val MAIN_FILE_NAME = "mainFile"
  val BUILD_BEFORE_PROP_NAME = "buildBefore"
  val LAUNCH_REQUEST_TYPE = "launch"

  val DEFAULT_CONFIG_NAMES: util.List[String] = util.Arrays.asList(TYPE_PROP_NAME, REQUEST_PROP_NAME, NAME_PROP_NAME)
}

case class LaunchConfigurationProperty(
                                        //The name of the property
                                        name: String,
                                        //The value of the property String, Boolean Number
                                        value: String
                                      )


case class OpenTextDocumentParams(
                                   // The uri of the file
                                   uri: String,
                                   // An optional range to select
                                   @Nullable range: org.eclipse.lsp4j.Range = null,
                                 )


case class EditorDecoration(
                             range: org.eclipse.lsp4j.Range,
                             text: String,
                             color: String
                           )

case class EditorDecorationsParams(
                                    documentUri: String,
                                    decorations: java.util.List[EditorDecoration]
                                  )

case class WeaveInputBoxParams(
                                //Set title of the input box window.
                                @Nullable title: String = null,
                                // The value to prefill in the input box
                                @Nullable value: String = null,
                                // The text to display underneath the input box.
                                @Nullable prompt: String = null,
                                // An optional string to show as place holder in the input box to guide the user what to type.
                                @Nullable placeholder: String = null,
                                // Set to `true` to show a password prompt that will not show the typed value.
                                @Nullable password: java.lang.Boolean = null,
                                // Set to `true` to keep the input box open when focus moves to another
                                // part of the editor or to another window.
                                @Nullable ignoreFocusOut: java.lang.Boolean = null,
                                @Nullable valueSelection: Array[Int] = null,
                                // Set list a custom buttons to appear on the quick pick.
                                @Nullable buttons: java.util.List[WeaveButton] = null,
                                //Set number if you are doing a step by step wizard.
                                @Nullable step: java.lang.Integer = null,
                                //Set number of total steps if you are doing a step by step wizard.
                                @Nullable totalSteps: java.lang.Integer = null
                              )

case class WeaveInputBoxResult(
                                // value=null when cancelled=true
                                @Nullable value: String = null,
                                @Nullable cancelled: java.lang.Boolean = null,
                                @Nullable buttonPressedId: String = null
                              )

case class WeaveQuickPickParams(
                                 items: java.util.List[WeaveQuickPickItem],
                                 //Set title of the quick pick window.
                                 @Nullable title: String = null,
                                 // An optional flag to include the description when filtering the picks.
                                 @Nullable matchOnDescription: java.lang.Boolean = null,
                                 // An optional flag to include the detail when filtering the picks.
                                 @Nullable matchOnDetail: java.lang.Boolean = null,
                                 // An optional string to show as place holder in the input box to guide the user what to pick on.
                                 @Nullable placeHolder: String = null,
                                 // Set to `true` to keep the picker open when focus moves to another part of the editor or to another window.
                                 @Nullable ignoreFocusOut: java.lang.Boolean = null,
                                 // Set list a custom buttons to appear on the quick pick.
                                 @Nullable buttons: java.util.List[WeaveButton] = null,
                                 // Set to `true` to allow user to select many of the options.
                                 @Nullable canSelectMany: java.lang.Boolean = null,
                                 //Set step number if you are doing a step by step wizard.
                                 @Nullable step: java.lang.Integer = null,
                                 //Set number of total steps if you are doing a step by step wizard.
                                 @Nullable totalSteps: java.lang.Integer = null,
                               )

case class WeaveQuickPickResult(
                                 // value=null when cancelled=true
                                 @Nullable buttonPressedId: String = null,
                                 @Nullable itemsId: java.util.List[String] = null,
                                 @Nullable cancelled: java.lang.Boolean = null
                               )

case class WeaveQuickPickItem(
                               id: String,
                               // A human readable string which is rendered prominent.
                               label: String,
                               // A human readable string which is rendered less prominent.
                               @Nullable description: String = null,
                               // A human readable string which is rendered less prominent.
                               @Nullable detail: String = null,
                               // Always show this item.
                               @Nullable alwaysShow: java.lang.Boolean = null,
                               // Set to true if this option is picked by default (only if it's a multiplepick)
                               @Nullable picked: java.lang.Boolean = null
                             )

case class OpenWindowsParams(
                              uri: String,
                              openNewWindow: java.lang.Boolean
                            )

case class WeaveOpenDialogParams(
                                canSelectFiles: java.lang.Boolean,
                                canSelectFolders: java.lang.Boolean,
                                canSelectMany: java.lang.Boolean,
                                defaultUri: String,
                                openLabel: String,
                                title: String
                                )

case class WeaveButton(
                        id: String,
                        iconPath: ThemeIconPath,
                        @Nullable tooltip: String = null
                      )

sealed trait ThemeIconPath {
  def iconType: String
}

//Icon Uris for dark and light theme only.
case class DarkLightIcon(dark: String,
                         light: String, override val iconType: String = "DARK-LIGHT-ICON") extends ThemeIconPath

//Unified Icon for every theme URI
case class IconUri(uri: String, override val iconType: String = "URI-ICON") extends ThemeIconPath

//Icon provided by the client side matched by id.
case class ThemeIcon(id: String, override val iconType: String = "THEME-ICON") extends ThemeIconPath

case class ShowScenariosParams(
                                //The name identifier of the Document
                                nameIdentifier: String,
                                //Al the scenarios that this document has
                                @Nullable scenarios: java.util.List[WeaveScenario] = null)

case class WeaveScenario(
                          active: java.lang.Boolean = false,
                          name: String,
                          uri: String,
                          //All the uri of all the input files
                          @Nullable inputsUri: Array[SampleInput] = null,
                          //TODO this should be a single url as just one output is accepted
                          @Nullable outputsUri: String = null
                        )

//TODO use something like this for inputs
case class SampleInput(uri: String, name: String)
