package org.mule.weave.lsp.project

import com.google.gson.JsonObject
import org.eclipse.lsp4j.InitializeParams
import org.mule.weave.lsp.project.Settings._
import org.mule.weave.lsp.project.events.SettingsChangedEvent
import org.mule.weave.lsp.utils.EventBus
import org.mule.weave.lsp.utils.URLUtils

import java.io.File
import scala.collection.mutable.ArrayBuffer

case class Project(url: Option[String], settings: ProjectSettings) {
  private val mayBeHome = url.flatMap(uri => URLUtils.toFile(uri))

  @transient
  private var projectStarted: Boolean = false

  def home(): File = {
    mayBeHome.get
  }

  def hasHome: Boolean = {
    mayBeHome.isDefined
  }

  def isStarted: Boolean = {
    projectStarted
  }

  def markStarted(): Unit = projectStarted = true
}

case class ProjectSettings(eventBus: EventBus,
                           wlangVersion: Setting = Setting(WLANG_VERSION_PROP_NAME, DEFAULT_VERSION),
                           languageLevelVersion: Setting = Setting(LANGUAGE_LEVEL_PROP_NAME, DEFAULT_VERSION),
                           validationLevel: Setting = Setting(VALIDATION_LEVEL_PROP_NAME, TYPE_LEVEL),
                           batVersion: Setting = Setting(BAT_VERSION_PROP_NAME, DEFAULT_BAT_VERSION),
                           batWrapperVersion: Setting = Setting(BAT_WRAPPER_VERSION_PROP_NAME, DEFAULT_BAT_WRAPPER_VERSION),
                           previewTimeout: Setting = Setting(PREVIEW_TIMEOUT, PREVIEW_TIMEOUT_VALUE),
                           useLiteralOnInsertType: Setting = Setting(USE_LITERAL_ON_INSERT, false.toString),
                           scaffoldingElementNumber: Setting = Setting(SCAFFOLDING_REPEATED_ELEMENT, 5.toString)) {
  val PROPERTIES =
    Seq(
      wlangVersion,
      languageLevelVersion,
      validationLevel,
      batVersion,
      batWrapperVersion,
      previewTimeout,
      useLiteralOnInsertType,
      scaffoldingElementNumber
    )

  def load(settings: AnyRef): Array[String] = {
    val allSettings: JsonObject = settings.asInstanceOf[JsonObject]
    val modifiedProps = new ArrayBuffer[String]()
    if (allSettings != null) {
      val weaveSettings = allSettings.getAsJsonObject("data-weave")

      if (weaveSettings != null) {
        PROPERTIES.foreach(prop => {
          if (prop.updateValue(weaveSettings)) {
            modifiedProps.+=(prop.settingName)
          }
        })
      }
    }
    modifiedProps.toArray
  }

  def update(settings: AnyRef): Unit = {
    val modified: Array[String] = load(settings)
    if (modified.nonEmpty) {
      eventBus.fire(new SettingsChangedEvent(modified))
    }
  }
}

object Settings {
  val DEFAULT_VERSION = "2.5.0"
  val DEFAULT_BAT_VERSION = "1.24.36"
  val DEFAULT_BAT_WRAPPER_VERSION = "1.0.58"
  val PREVIEW_TIMEOUT_VALUE: String = 50000.toString
  val DEFAULT_BAT_HOME = ".bat"
  val TYPE_LEVEL = "type"
  val SCOPE_LEVEL = "scope"
  val PARSE_LEVEL = "parse"
  val WLANG_VERSION_PROP_NAME = "wlangVersion"
  val LANGUAGE_LEVEL_PROP_NAME = "languageLevel"
  val VALIDATION_LEVEL_PROP_NAME = "validationLevel"
  val BAT_VERSION_PROP_NAME = "batVersion"
  val BAT_WRAPPER_VERSION_PROP_NAME = "batWrapperVersion"
  val PREVIEW_TIMEOUT = "previewTimeout"
  val USE_LITERAL_ON_INSERT = "userLiteralOnInsertAction"
  val SCAFFOLDING_REPEATED_ELEMENT = "scaffoldingRepeatedElement"

  def isTypeLevel(settings: ProjectSettings): Boolean = settings.validationLevel.value() == TYPE_LEVEL

  def isScopeLevel(settings: ProjectSettings): Boolean = settings.validationLevel.value() == SCOPE_LEVEL

  def isParseLevel(settings: ProjectSettings): Boolean = settings.validationLevel.value() == PARSE_LEVEL
}

case class Setting(settingName: String, initialValue: String) {

  private var valueHolder: String = initialValue

  def value(): String = valueHolder

  def intValue(): Int = valueHolder.toInt

  def booleanValue(): Boolean = valueHolder.toBoolean

  def updateValue(v: JsonObject): Boolean = {
    Option(v.get(settingName)) match {
      case Some(value) =>
        val newValue = value.getAsString
        if (!newValue.equals(valueHolder)) {
          valueHolder = newValue
          true
        } else {
          false
        }
      case None =>
        false
    }
  }
}

object Project {
  def create(initParams: InitializeParams, eventBus: EventBus): Project = {
    val rootPath: Option[String] = Option(initParams.getRootUri).orElse(Option(initParams.getRootPath))
    val settings: ProjectSettings = ProjectSettings(eventBus)
    settings.load(initParams.getInitializationOptions)
    Project(rootPath, settings)
  }
}
