package org.mule.weave.lsp.project.impl.maven

import org.apache.commons.codec.digest.DigestUtils
import org.jboss.shrinkwrap.resolver.impl.maven.embedded.WeaveBinaryDownloader
import org.jboss.shrinkwrap.resolver.impl.maven.embedded.WeaveFileExtractor
import org.mule.weave.lsp.services.ClientLogger

import java.io.File
import java.io.FileFilter
import java.io.FileInputStream
import java.io.IOException
import java.net.URL
import java.nio.file.Paths

class MavenBinaryDistributionInstaller(logger: ClientLogger) {
  private val MAVEN_3_BASE_URL: String = "https://archive.apache.org/dist/maven/maven-3/3.6.3/binaries/apache-maven-3.6.3-bin.tar.gz"
  private val ARQUILLIAN_DIR: String = System.getProperty("user.home") + File.separator + ".arquillian" + File.separator + "resolver" + File.separator + "maven"

  def mavenHome(): Option[File] = {
    synchronized {
      val mavenDistribution = new URL(MAVEN_3_BASE_URL)
      val mavenDir = prepareMavenDir()
      val downloaded = WeaveBinaryDownloader.download(mavenDir, mavenDistribution)
      val maybeDownloadedZipMd5hash = getMd5hash(downloaded)
      var binDirectory: File = null
      if (maybeDownloadedZipMd5hash.isDefined) {
        val downloadedZipMd5hash = maybeDownloadedZipMd5hash.get
        val destinationDir = Paths.get(ARQUILLIAN_DIR + File.separator + "extracted", downloadedZipMd5hash).toFile
        val withExtractedDir = WeaveFileExtractor.extract(downloaded, destinationDir)
        binDirectory = retrieveBinDirectory(withExtractedDir)
      }
      Option(binDirectory)
    }
  }

  private def prepareMavenDir(): File = {
    val mavenDir = new File(ARQUILLIAN_DIR)
    if (!mavenDir.exists) {
      mavenDir.mkdirs
    }
    mavenDir
  }

  private def getMd5hash(downloaded: File): Option[String] = {
    var fis: FileInputStream = null
    var hash: String = null
    try {
      fis = new FileInputStream(downloaded)
      hash = DigestUtils.md5Hex(fis)
    } catch {
      case e: IOException =>
        logger.logWarning("A problem occurred when md5 hash of a file " + downloaded + " was being retrieved:\n" + e.getMessage)
    } finally {
      if (fis != null) {
        try {
          fis.close()
        } catch {
          case e: IOException =>
            logger.logWarning("A problem occurred when FileInputStream of a file " + downloaded + "was being closed:\n" + e.getMessage)
        }
      }
    }
    Option(hash)
  }

  private def retrieveBinDirectory(uncompressed: File): File = {
    val extracted = uncompressed.listFiles(new FileFilter {
      override def accept(pathname: File): Boolean = {
        pathname.isDirectory
      }
    })
    if (extracted.isEmpty) {
      throw new IllegalArgumentException("No directory has been extracted from the archive: " + uncompressed)
    }
    if (extracted.length > 1) {
      throw new IllegalArgumentException("More than one directory has been extracted from the archive: " + uncompressed)
    }
    extracted(0)
  }
}

object MavenBinaryDistributionInstaller {
  def apply(logger: ClientLogger): MavenBinaryDistributionInstaller = new MavenBinaryDistributionInstaller(logger)
}
