package org.mule.weave.lsp.actions

import org.eclipse.lsp4j
import org.eclipse.lsp4j.CodeAction
import org.eclipse.lsp4j.CodeActionKind
import org.eclipse.lsp4j.CodeActionParams
import org.eclipse.lsp4j.Position
import org.mule.weave.lsp.commands.CommandProvider
import org.mule.weave.lsp.commands.ExtractConstantCommand
import org.mule.weave.lsp.commands.ExtractFunctionCommand
import org.mule.weave.lsp.commands.ExtractVariableCommand
import org.mule.weave.lsp.project.components.ContextMetadata
import org.mule.weave.lsp.services.DataWeaveToolingService
import org.mule.weave.v2.editor.WeaveDocumentToolingService
import org.mule.weave.v2.parser.ast.AstNodeHelper

class RefactorActionProvider(toolingService: DataWeaveToolingService, commandProvider: CommandProvider) extends CodeActionProvider {
  override def doHandles(params: CodeActionParams, maybeContext: Option[ContextMetadata]): Boolean = {
    val documentToolingService = toolingService.openDocument(params.getTextDocument.getUri, maybeContext, withExpectedOutput = true)
    val range: lsp4j.Range = params.getRange
    val startPosition: Position = range.getStart
    val endPosition: Position = range.getEnd

    val startOffset: Int = toOffset(documentToolingService, startPosition)
    val endOffset: Int = toOffset(documentToolingService, endPosition)
    val maybeNode = documentToolingService.nodeAt(startOffset, endOffset)
    maybeNode match {
      case Some(nodeToExtract) =>
        AstNodeHelper.isExpressionNode(nodeToExtract) || AstNodeHelper.isWeaveTypeNode(nodeToExtract)
      case None => false
    }
  }

  private def toOffset(documentToolingService: WeaveDocumentToolingService, startPosition: Position) = {
    documentToolingService.offsetOf(startPosition.getLine, startPosition.getCharacter)
  }

  override def doActions(params: CodeActionParams, maybeContext: Option[ContextMetadata]): Array[CodeAction] = {
    val documentToolingService = toolingService.openDocument(params.getTextDocument.getUri, maybeContext, withExpectedOutput = true)
    val range: lsp4j.Range = params.getRange
    val startPosition: Position = range.getStart
    val endPosition: Position = range.getEnd
    val startOffset: Int = toOffset(documentToolingService, startPosition)
    val endOffset: Int = toOffset(documentToolingService, endPosition)

    val extractVariable = new CodeAction("Extract Variable (Local Scope)")
    extractVariable.setCommand(ExtractVariableCommand.createCommand(startOffset, endOffset, params.getTextDocument.getUri, commandProvider))
    extractVariable.setKind(CodeActionKind.RefactorExtract + ".variable")

    val extractConstant = new CodeAction("Extract Variable (Global Scope)")
    extractConstant.setCommand(ExtractConstantCommand.createCommand(startOffset, endOffset, params.getTextDocument.getUri, commandProvider))
    extractConstant.setKind(CodeActionKind.RefactorExtract + ".constant")

    val extractFunction = new CodeAction("Extract Function")
    extractFunction.setCommand(ExtractFunctionCommand.createCommand(startOffset, endOffset, params.getTextDocument.getUri, commandProvider))
    extractFunction.setKind(CodeActionKind.RefactorExtract + ".function")
    Array(extractVariable, extractConstant, extractFunction)
  }
}
