package org.mule.weave.lsp.project.impl.simple

import org.jboss.shrinkwrap.resolver.api.maven.Maven
import org.jboss.shrinkwrap.resolver.api.maven.MavenFormatStage
import org.jboss.shrinkwrap.resolver.api.maven.MavenResolvedArtifact
import org.mule.weave.lsp.agent.WeaveAgentService
import org.mule.weave.lsp.project.Project
import org.mule.weave.lsp.project.ProjectKind
import org.mule.weave.lsp.project.ProjectKindContext
import org.mule.weave.lsp.project.ProjectKindDetector
import org.mule.weave.lsp.project.Settings
import org.mule.weave.lsp.project.components
import org.mule.weave.lsp.project.components.BuildManager
import org.mule.weave.lsp.project.components.DefaultSampleDataManager
import org.mule.weave.lsp.project.components.DependencyArtifact
import org.mule.weave.lsp.project.components.MetadataProvider
import org.mule.weave.lsp.project.components.ModuleStructure
import org.mule.weave.lsp.project.components.NoBuildManager
import org.mule.weave.lsp.project.components.ProjectDependencyManager
import org.mule.weave.lsp.project.components.ProjectStructure
import org.mule.weave.lsp.project.components.RootKind
import org.mule.weave.lsp.project.components.RootStructure
import org.mule.weave.lsp.project.components.SampleBaseMetadataProvider
import org.mule.weave.lsp.project.components.SampleDataManager
import org.mule.weave.lsp.project.events.DependencyArtifactRemovedEvent
import org.mule.weave.lsp.project.events.DependencyArtifactResolvedEvent
import org.mule.weave.lsp.project.events.OnSettingsChanged
import org.mule.weave.lsp.project.events.SettingsChangedEvent
import org.mule.weave.lsp.project.impl.simple.WeaveArtifacts.IO_VERSION
import org.mule.weave.lsp.project.impl.simple.WeaveArtifacts.createWeaveArtifactId
import org.mule.weave.lsp.services.ClientLogger
import org.mule.weave.lsp.services.WeaveScenarioManagerService
import org.mule.weave.lsp.utils.EventBus
import org.mule.weave.lsp.utils.JavaLoggerForwarder.interceptLog
import org.mule.weave.lsp.utils.WeaveDirectoryUtils

import java.io.File
import scala.collection.immutable

class SimpleProjectKind(
                         project: Project,
                         logger: ClientLogger,
                         eventBus: EventBus,
                         weaveAgentService: WeaveAgentService,
                         weaveScenarioManagerService: WeaveScenarioManagerService,
                         sampleDataFolder: File = WeaveDirectoryUtils.getDWHome(),
                         dependencies: Seq[MavenArtifactIdResolver] = WeaveArtifacts.BASIC_WITH_JAVA_MODULES
                       ) extends ProjectKind {
  private val simpleDependencyManager = new SimpleDependencyManager(project, logger, eventBus, dependencies)
  private val defaultSampleDataManager = new DefaultSampleDataManager(sampleDataFolder)
  private val sampleBaseMetadataProvider = new SampleBaseMetadataProvider(weaveAgentService, eventBus, weaveScenarioManagerService)

  override def name(): String = "DW `Simple`"

  override def structure(): ProjectStructure = {
    if (project.hasHome) {
      val mainSourceFolder = new File(project.home(), "src")
      val mainRoot = RootStructure(RootKind.MAIN, Array(mainSourceFolder), Array.empty, mainSourceFolder)
      val modules = Array(ModuleStructure(project.home().getName, Array(mainRoot)))
      components.ProjectStructure(modules)
    } else {
      components.ProjectStructure(Array.empty)
    }
  }

  override def dependencyManager(): ProjectDependencyManager = {
    simpleDependencyManager
  }

  override def buildManager(): BuildManager = {
    NoBuildManager
  }

  override def sampleDataManager(): SampleDataManager = {
    defaultSampleDataManager
  }

  override def metadataProvider(): Option[MetadataProvider] = {
    Some(sampleBaseMetadataProvider)
  }
}

class SpellProjectKindDetector() extends ProjectKindDetector {
  override def supports(context: ProjectKindContext): Boolean = {
    new File(context.project.home(), "src").exists()
  }

  override def createKind(context: ProjectKindContext): ProjectKind = {
    new SimpleProjectKind(
      context.project,
      context.clientLogger,
      context.eventBus,
      context.weaveAgentService,
      context.weaveScenarioManagerService,
      context.project.home(),
      WeaveArtifacts.NATIVE_MODULES
    )
  }
}

class SimpleDependencyManager(project: Project, logger: ClientLogger, eventBus: EventBus, dependencies: Seq[MavenArtifactIdResolver], dependantSettings: Seq[String] = Seq(Settings.WLANG_VERSION_PROP_NAME)) extends ProjectDependencyManager {

  var dependenciesArray: Array[DependencyArtifact] = Array.empty

  eventBus.register(SettingsChangedEvent.SETTINGS_CHANGED, new OnSettingsChanged {
    override def onSettingsChanged(modifiedSettingsName: Array[String]): Unit = {
      if (modifiedSettingsName.exists((p) => dependantSettings.contains(p))) {
        reloadDependencies()
      }
    }
  })

  override def start(): Unit = {
    reloadDependencies()
  }

  override def reload(): Unit = {}

  protected def reloadDependencies(): Unit = {
    val allArtifacts: Array[DependencyArtifact] = loadNewArtifacts()
    dependenciesArray = allArtifacts
    eventBus.fire(new DependencyArtifactResolvedEvent(dependenciesArray))
  }

  protected def loadNewArtifacts(): Array[DependencyArtifact] = {
    dependencies.flatMap((gav) => {
      resolveDependency(gav.artifactName(project))
    }).toArray
  }

  protected def resolveDependency(artifactID: String): Array[DependencyArtifact] = {
    interceptLog(logger) {
      val mavenFormatStage: MavenFormatStage = Maven.configureResolver()
        .withMavenCentralRepo(true)
        .withRemoteRepo("mule-releases", "https://repository-master.mulesoft.org/nexus/content/repositories/releases", "default")
        .withRemoteRepo("mule-snapshots", "https://repository-master.mulesoft.org/nexus/content/repositories/snapshots", "default")
        .resolve(artifactID)
        .withTransitivity
      val dependencies: immutable.Iterable[MavenResolvedArtifact] =
        mavenFormatStage
          .asResolvedArtifact()
          .groupBy(_.getCoordinate.toCanonicalForm)
          .map(_._2.head)
      if (dependenciesArray.nonEmpty) {
        eventBus.fire(new DependencyArtifactRemovedEvent(dependenciesArray))
      }
      dependencies.map((a) => {
        DependencyArtifact(a.getCoordinate.toCanonicalForm, a.asFile())
      }).toArray
    }
  }

  override def languageLevel(): String = {
    project.settings.wlangVersion.value()
  }

  override def dependencies(): Array[DependencyArtifact] = {
    dependenciesArray
  }
}

trait MavenArtifactIdResolver {
  def artifactName(project: Project): String
}


object RuntimeArtifactIdResolver extends MavenArtifactIdResolver {
  override def artifactName(project: Project): String = {
    createWeaveArtifactId("runtime", project.settings.wlangVersion.value())
  }
}

object CoreModulesArtifactIdResolver extends MavenArtifactIdResolver {
  override def artifactName(project: Project): String = {
    createWeaveArtifactId("core-modules", project.settings.wlangVersion.value())
  }
}

object YamlModuleArtifactIdResolver extends MavenArtifactIdResolver {
  override def artifactName(project: Project): String = {
    createWeaveArtifactId("yaml-module", project.settings.wlangVersion.value())
  }
}

object JavaModuleArtifactIdResolver extends MavenArtifactIdResolver {
  override def artifactName(project: Project): String = {
    createWeaveArtifactId("java-module", project.settings.wlangVersion.value())
  }
}

object NDJsonModuleArtifactIdResolver extends MavenArtifactIdResolver {
  override def artifactName(project: Project): String = {
    createWeaveArtifactId("ndjson-module", project.settings.wlangVersion.value())
  }
}

object FileModuleArtifactIdResolver extends MavenArtifactIdResolver {
  override def artifactName(project: Project): String = {
    createWeaveArtifactId("file-module", IO_VERSION)
  }
}

object HttpModuleArtifactIdResolver extends MavenArtifactIdResolver {
  override def artifactName(project: Project): String = {
    createWeaveArtifactId("http-netty-module", IO_VERSION)
  }
}

object ProcessModuleArtifactIdResolver extends MavenArtifactIdResolver {
  override def artifactName(project: Project): String = {
    createWeaveArtifactId("process-module", IO_VERSION)
  }
}

object BATArtifactIdResolver extends MavenArtifactIdResolver {
  override def artifactName(project: Project): String = {
    "com.mulesoft.bat:bdd-core:" + project.settings.batVersion.value()
  }
}

object WeaveArtifacts {

  val IO_VERSION = "0.9.0"

  val BASIC_MODULES: Seq[MavenArtifactIdResolver] = Seq(RuntimeArtifactIdResolver, CoreModulesArtifactIdResolver, NDJsonModuleArtifactIdResolver, YamlModuleArtifactIdResolver)

  val BASIC_WITH_JAVA_MODULES: Seq[MavenArtifactIdResolver] = BASIC_MODULES :+ JavaModuleArtifactIdResolver

  val BAT_MODULES: Seq[MavenArtifactIdResolver] = BASIC_MODULES :+ BATArtifactIdResolver

  val NATIVE_MODULES: Seq[MavenArtifactIdResolver] = BASIC_MODULES ++ Seq(FileModuleArtifactIdResolver, HttpModuleArtifactIdResolver, ProcessModuleArtifactIdResolver)

  def createWeaveArtifactId(module: String, version: String): String = {
    s"org.mule.weave:${module}:${version}"
  }
}

