package org.mule.weave.lsp.services

import org.apache.commons.io.FileUtils
import org.apache.commons.io.FilenameUtils
import org.eclipse.lsp4j.ApplyWorkspaceEditParams
import org.eclipse.lsp4j.ApplyWorkspaceEditResponse
import org.eclipse.lsp4j.CreateFile
import org.eclipse.lsp4j.DeleteFile
import org.eclipse.lsp4j.Position
import org.eclipse.lsp4j.ResourceOperation
import org.eclipse.lsp4j.TextDocumentEdit
import org.eclipse.lsp4j.TextEdit
import org.eclipse.lsp4j.VersionedTextDocumentIdentifier
import org.eclipse.lsp4j.WorkspaceEdit
import org.eclipse.lsp4j.jsonrpc.messages.Either
import org.mule.weave.lsp.extension.client
import org.mule.weave.lsp.extension.client.SampleInput
import org.mule.weave.lsp.extension.client.ShowScenariosParams
import org.mule.weave.lsp.extension.client.WeaveLanguageClient
import org.mule.weave.lsp.extension.client.WeaveScenario
import org.mule.weave.lsp.project.Project
import org.mule.weave.lsp.project.ProjectKind
import org.mule.weave.lsp.project.components.SampleDataManager
import org.mule.weave.lsp.project.components.Scenario
import org.mule.weave.lsp.services.events.ActiveScenarioChangedEvent
import org.mule.weave.lsp.services.events.DocumentFocusChangedEvent
import org.mule.weave.lsp.services.events.OnDocumentFocused
import org.mule.weave.lsp.utils.URLUtils.toLSPUrl
import org.mule.weave.lsp.utils.WeaveDirectoryUtils.toFolderName
import org.mule.weave.lsp.utils._
import org.mule.weave.v2.editor.VirtualFile
import org.mule.weave.v2.editor.VirtualFileSystem
import org.mule.weave.v2.parser.ast.AstNodeHelper
import org.mule.weave.v2.parser.ast.functions.FunctionCallNode
import org.mule.weave.v2.parser.ast.functions.FunctionCallParametersNode
import org.mule.weave.v2.parser.ast.structure.ArrayNode
import org.mule.weave.v2.parser.ast.variables.NameIdentifier
import org.mule.weave.v2.parser.ast.variables.VariableReferenceNode
import org.mule.weave.v2.sdk.NameIdentifierHelper.toWeaveFilePath
import org.mule.weave.v2.utils.StringHelper.toStringTransformer

import java.io.File
import java.nio.charset.Charset
import java.util
import scala.collection.JavaConverters.asScalaIteratorConverter
import scala.collection.JavaConverters.seqAsJavaListConverter
import scala.collection.mutable


class WeaveScenarioManagerService(dataWeaveToolingService: DataWeaveToolingService, weaveLanguageClient: WeaveLanguageClient, val virtualFileSystem: VirtualFileSystem) extends ToolingService {


  private var eventBus: EventBus = _
  protected var projectKind: ProjectKind = _

  /**
    * The active scenario for each mapping
    */
  var activeScenarios: mutable.HashMap[NameIdentifier, Scenario] = mutable.HashMap()

  def mapScenarios(maybeActiveScenario: Option[Scenario], allScenarios: Array[Scenario]): util.List[client.WeaveScenario] = {
    val defaultScenarioName = maybeActiveScenario.map(_.name).getOrElse("")
    val scenarios: Array[WeaveScenario] = allScenarios.map(scenario => {
      val inputsList: Array[SampleInput] = scenario.inputs()
      val expectedOrNull: String = scenario.expected().map((file) => URLUtils.toLSPUrl(file)).orNull
      WeaveScenario(scenario.name.equals(defaultScenarioName), scenario.name, URLUtils.toLSPUrl(scenario.file), inputsList, expectedOrNull)
    })
    scenarios.toList.asJava
  }

  override def init(projectKind: ProjectKind, eventBus: EventBus): Unit = {
    this.projectKind = projectKind
    this.eventBus = eventBus
    eventBus.register(DocumentFocusChangedEvent.DOCUMENT_FOCUS_CHANGED, new OnDocumentFocused {

      override def onDocumentFocused(vf: VirtualFile): Unit = {
        val resources: Array[File] = WeaveDirectoryUtils.wtfResourcesTestFolder(projectKind.structure())
        val isPartOfTheTestResources = !resources.exists((f) => URLUtils.isChildOf(vf.url(), f))
        if (URLUtils.isSupportedEditableScheme(vf.url()) && isPartOfTheTestResources) {
          notifyAllScenarios(vf)
        }
      }
    })
  }

  def listScenarios(nameIdentifier: NameIdentifier): Array[Scenario] = {
    projectKind
      .sampleDataManager()
      .listScenarios(nameIdentifier)
  }

  def searchScenarioByName(nameIdentifier: NameIdentifier, scenarioName: String): Option[Scenario] = {
    projectKind.sampleDataManager().searchScenarioByName(nameIdentifier, scenarioName)
  }

  def activeScenario(nameIdentifier: NameIdentifier): Option[Scenario] = {
    activeScenarios.get(nameIdentifier)
      .orElse({
        val scenarios: Array[Scenario] = listScenarios(nameIdentifier)
        if (scenarios != null && scenarios.nonEmpty) {
          val maybeFirstScenario: Option[Scenario] =
            scenarios
              .find(s => s.name.equals(Scenario.PLAYGROUND_SCENARIO)) //Try to find the default scenario
              .orElse(scenarios.headOption)
          maybeFirstScenario.foreach((scenario) => activeScenarios.put(nameIdentifier, scenario))
          maybeFirstScenario
        } else {
          None
        }
      })
  }

  def copyScenarioTo(mappingIdentifier: NameIdentifier, oldScenario: Scenario, scenarioName: String): Unit = {
    val inputs = oldScenario.inputs()
    if (inputs.isEmpty) {
      createScenario(mappingIdentifier, scenarioName, WeaveScenarioManagerService.DEFAULT_INPUT, None)
    } else {
      inputs.foreach(input => {
        URLUtils.toFile(input.uri).map(oldInputFile => {
          val inputName = s"${input.name}.${FilenameUtils.getExtension(oldInputFile.getName)}"
          val content = FileUtils.readFileToString(oldInputFile, Charset.defaultCharset())
          createScenario(mappingIdentifier, scenarioName, inputName, Some(content))
        })
      })
    }
  }

  def setActiveScenario(nameIdentifier: NameIdentifier, nameOfTheScenario: String): Unit = {
    val maybeScenario = projectKind.sampleDataManager()
      .listScenarios(nameIdentifier)
      .find((scenario) => {
        scenario.name.equals(nameOfTheScenario)
      })

    maybeScenario.foreach((s) => {
      activeScenarios.put(nameIdentifier, s)
      notifyAllScenarios(nameIdentifier)
    })
    val virtualFiles: Iterator[VirtualFile] = virtualFileSystem.listFiles().asScala
    virtualFiles.find(vf => vf.getNameIdentifier.equals(nameIdentifier)).foreach(vf => eventBus.fire(new ActiveScenarioChangedEvent(vf)))
  }

  def deleteScenario(nameIdentifier: NameIdentifier, nameOfTheScenario: String): Unit = {
    projectKind.sampleDataManager()
      .searchScenarioByName(nameIdentifier, nameOfTheScenario)
      .foreach((s) => {
        FileUtils.deleteDirectory(s.file)
      })
    activeScenarios.get(nameIdentifier).foreach(activeScenario => {
      if (activeScenario.name == nameOfTheScenario) {
        activeScenarios.remove(nameIdentifier)
      }
    })
    notifyAllScenarios(nameIdentifier)
  }

  def createOrUpdateMappingTest(nameIdentifier: NameIdentifier, scenarioName: String, mimeType: String): Option[File] = {
    val maybeScenario = projectKind.sampleDataManager().searchScenarioByName(nameIdentifier, scenarioName)
    val maybeFile = maybeScenario
      .flatMap((scenario) => {
        doSaveTest(nameIdentifier, scenario, mimeType)
      })
    maybeFile

  }

  private def doSaveTest(nameIdentifier: NameIdentifier, scenario: Scenario, mimeType: String): Option[File] = {

    val testFile: File = new File(WeaveDirectoryUtils.wtfUnitDefaultTestSourceFolder(projectKind.structure())(0), s"${toFolderName(nameIdentifier.parent().getOrElse(NameIdentifier("")))}/${nameIdentifier.localName()}Test.dwl")

    val scenarioTemplate =
      s"""
         |    "Assert ${scenario.name}" in do {
         |        evalPath("${toWeaveFilePath(nameIdentifier, "/").drop(1)}", inputsFrom("${toFolderName(nameIdentifier)}/${scenario.name}"),"${mimeType}") must
         |                  equalTo(outputFrom("${toFolderName(nameIdentifier)}/${scenario.name}"))
         |    }""".stripMarginAndNormalizeEOL

    val testTemplate =
      s"""%dw ${projectKind.getWeaveVersion()}
         |import * from dw::test::Tests
         |import * from dw::test::Asserts
         |---
         |"Test ${nameIdentifier.name}" describedBy [$scenarioTemplate
         |]
         |""".stripMarginAndNormalizeEOL

    def appendScenarioToExistingTest: Option[File] = {

      val documentToolingService = dataWeaveToolingService.openDocument(toLSPUrl(testFile))
      val maybePosition = documentToolingService.ast().flatMap(
        AstNodeHelper.find(_, {
          case FunctionCallNode(VariableReferenceNode(NameIdentifier("describedBy", _)), FunctionCallParametersNode(_), _) => true
          case _ => false
        }).flatMap({
          case FunctionCallNode(VariableReferenceNode(NameIdentifier("describedBy", _)), FunctionCallParametersNode(Seq(_, array: ArrayNode)), _) if array.children().nonEmpty =>
            Some(LSPConverters.toPosition(array.children().last.location().endPosition))
          case _ => None
        }))

      maybePosition.flatMap(p =>
        if (applyEdits(editFileCmd(testFile, p, p, s",$scenarioTemplate")).isApplied) {
          notifyAllScenarios(nameIdentifier)
          Some(testFile)
        } else None)
    }

    def createTestWithFirstScenario: Option[File] = {
      val docStart = new Position(0, 0)

      val response = applyEdits(
        createFileCmd(testFile),
        editFileCmd(testFile, docStart, docStart, testTemplate)
      )

      if (response.isApplied) {
        notifyAllScenarios(nameIdentifier)
        Some(testFile)
      } else {
        None
      }
    }

    if (testFile.exists()) {
      appendScenarioToExistingTest
    } else {
      createTestWithFirstScenario
    }
  }

  def saveOutput(nameIdentifier: NameIdentifier, nameOfTheScenario: String, outputName: String, newContent: String): Option[File] = {
    val maybeScenario = projectKind.sampleDataManager().searchScenarioByName(nameIdentifier, nameOfTheScenario)
    val maybeFile = maybeScenario
      .flatMap((scenario) => {
        doSaveOutput(nameIdentifier, outputName, scenario.file, newContent)
      })
    maybeFile

  }

  def deleteOutput(nameIdentifier: NameIdentifier, nameOfTheScenario: String, outputUrl: String): Unit = {
    projectKind.sampleDataManager().searchScenarioByName(nameIdentifier, nameOfTheScenario)
      .foreach((_) => {
        URLUtils.toFile(outputUrl)
          .foreach((f) => f.delete())
      })

    notifyAllScenarios(nameIdentifier)
  }

  def deleteInput(nameIdentifier: NameIdentifier, nameOfTheScenario: String, inputUrl: String): Unit = {
    projectKind.sampleDataManager().searchScenarioByName(nameIdentifier, nameOfTheScenario)
      .foreach((_) => {
        URLUtils.toFile(inputUrl)
          .foreach((f) => f.delete())
      })

    notifyAllScenarios(nameIdentifier)
  }

  def createScenario(nameIdentifier: NameIdentifier, nameOfTheScenario: String, inputFileName: String, maybeInputContent: Option[String] = None): Option[File] = {
    val sampleDataManager: SampleDataManager = projectKind.sampleDataManager()
    val sampleContainer: File = sampleDataManager.createSampleDataFolderFor(nameIdentifier)
    val scenario: File = new File(sampleContainer, nameOfTheScenario)
    val content: Option[String] = maybeInputContent.orElse(InputScaffoldingFactory.create(inputFileName).map(_.template()))
    doCreateInput(nameIdentifier, inputFileName, content, scenario)
  }

  protected def doCreateInput(nameIdentifier: NameIdentifier, inputFileName: String, maybeInputContent: Option[String], scenario: File): Option[File] = {
    val inputFile: File = inputOf(scenario, inputFileName)

    val edits = new util.ArrayList[Either[TextDocumentEdit, ResourceOperation]]()
    edits.add(createFileCmd(inputFile))

    if (maybeInputContent.isDefined) {
      val content: String = maybeInputContent.get
      val pos: Position = new Position(0, 0)
      edits.add(editFileCmd(inputFile, pos, pos, content))
    }
    val response: ApplyWorkspaceEditResponse = applyEdits(edits)
    notifyAllScenarios(nameIdentifier)
    if (response.isApplied) {
      Some(inputFile)
    } else {
      None
    }
  }

  private def deleteFileCmd(destinationFile: File): Either[TextDocumentEdit, ResourceOperation] = {
    val destinationFileUrl = toLSPUrl(destinationFile)
    val createFile = Either.forRight[TextDocumentEdit, ResourceOperation](new DeleteFile(destinationFileUrl))
    createFile
  }

  private def createFileCmd(destinationFile: File): Either[TextDocumentEdit, ResourceOperation] = {
    val destinationFileUrl = toLSPUrl(destinationFile)
    val createFile = Either.forRight[TextDocumentEdit, ResourceOperation](new CreateFile(destinationFileUrl))
    createFile
  }

  private def editFileCmd(destinationFile: File, startPos: Position, endPos: Position, newText: String): Either[TextDocumentEdit, ResourceOperation] = {
    val outputFileUrl: String = toLSPUrl(destinationFile)
    val textEdit = new TextEdit(new org.eclipse.lsp4j.Range(startPos, endPos), newText)
    val textDocumentEdit = new TextDocumentEdit(new VersionedTextDocumentIdentifier(outputFileUrl, 0), util.Arrays.asList(textEdit))
    val editFile = Either.forLeft[TextDocumentEdit, ResourceOperation](textDocumentEdit)
    editFile
  }

  private def applyEdits(edits: Either[TextDocumentEdit, ResourceOperation]*): ApplyWorkspaceEditResponse = {
    val editsList = util.Arrays.asList(edits: _*)
    applyEdits(editsList)
  }

  private def applyEdits(editsList: util.List[Either[TextDocumentEdit, ResourceOperation]]): ApplyWorkspaceEditResponse = {
    weaveLanguageClient.applyEdit(new ApplyWorkspaceEditParams(new WorkspaceEdit(editsList))).get()
  }

  private def doSaveOutput(nameIdentifier: NameIdentifier, outputName: String, scenario: File, newText: String): Option[File] = {
    val outputFile: File = outputOf(scenario, outputName)

    val docStart = new Position(0, 0)
    if (outputFile.exists()) {
      val result = applyEdits(deleteFileCmd(outputFile))
      if (!result.isApplied) {
        return None
      }
    }
    val response = applyEdits(
      createFileCmd(outputFile),
      editFileCmd(outputFile, docStart, docStart, newText)
    )
    if (response.isApplied) {
      notifyAllScenarios(nameIdentifier)
      Some(outputFile)
    } else {
      None
    }
  }


  /**
    * Gets the output file of a scenario
    */
  private def outputOf(scenario: File, outputName: String): File = {
    new File(scenario, outputName)
  }

  /**
    * Gets the input file of a scenario
    */
  protected def inputOf(scenario: File, inputName: String): File = {
    val inputs: File = WeaveDirectoryUtils.inputsFolder(scenario)
    new File(inputs, getVariablePath(inputName))
  }

  /**
    * This method allows supporting 'vars.varName'
    */
  private def getVariablePath(fileName: String): String = {
    val theBaseName: String = FilenameUtils.getBaseName(fileName)
    val extension: String = FilenameUtils.getExtension(fileName)
    val variablePath: String = theBaseName.replace('.', File.separatorChar) + "." + extension
    variablePath
  }

  private def notifyAllScenarios(vf: VirtualFile): Unit = {
    notifyAllScenarios(vf.getNameIdentifier)
  }

  private def notifyAllScenarios(nameIdentifier: NameIdentifier): Unit = {
    val maybeActiveScenario: Option[Scenario] = activeScenario(nameIdentifier)
    val allScenarios = projectKind.sampleDataManager().listScenarios(nameIdentifier)
    weaveLanguageClient.showScenarios(
      scenariosParam = ShowScenariosParams(
        nameIdentifier = nameIdentifier.toString(),
        scenarios = mapScenarios(maybeActiveScenario, allScenarios)
      )
    )
  }
}

object WeaveScenarioManagerService {
  val DEFAULT_INPUT = "payload.json"
}
