package org.mule.weave.lsp.commands

import org.mule.weave.lsp.agent.WeaveAgentService
import org.mule.weave.lsp.extension.client.WeaveLanguageClient
import org.mule.weave.lsp.jobs.JobManagerService
import org.mule.weave.lsp.project.Project
import org.mule.weave.lsp.project.ProjectKind
import org.mule.weave.lsp.services.AgentBasedDataFormatDescriptorProviderService
import org.mule.weave.lsp.services.ClientLogger
import org.mule.weave.lsp.services.DataWeaveTestService
import org.mule.weave.lsp.services.DataWeaveToolingService
import org.mule.weave.lsp.services.PreviewService
import org.mule.weave.lsp.services.ToolingService
import org.mule.weave.lsp.services.WeaveScenarioManagerService
import org.mule.weave.lsp.utils.EventBus
import org.mule.weave.lsp.vfs.ProjectVirtualFileSystem
import org.mule.weave.v2.editor.VirtualFileSystem

class CommandProvider(virtualFileSystem: VirtualFileSystem,
                      projectVirtualFileSystem: ProjectVirtualFileSystem,
                      clientLogger: ClientLogger,
                      languageClient: WeaveLanguageClient,
                      project: Project,
                      jobManagerService: JobManagerService,
                      weaveToolingService: DataWeaveToolingService,
                      weaveAgentService: WeaveAgentService,
                      scenariosManager: WeaveScenarioManagerService,
                      previewService: PreviewService,
                      dataWeaveTestService: DataWeaveTestService,
                      dataFormatsService: AgentBasedDataFormatDescriptorProviderService,
                      commandPrefix: Option[String]
                     ) extends  ToolingService {

  override def init(projectKind: ProjectKind, eventBus: EventBus): Unit = {
    initProjectCommands(projectKind)
  }

  def initProjectCommands(projectKind: ProjectKind) = {
    val commands = Seq(
//      TODO move to custom commands
      new RunBatTestCommand(clientLogger, projectKind, weaveToolingService, languageClient),
      new RunBatFolderTestCommand(clientLogger, projectKind, languageClient),
      new CreateTestCommand(projectKind, languageClient),
      new CreateMappingFileCommand(projectKind, languageClient),
      new CreateModuleFileCommand(projectKind, languageClient),
      new CreateIntegrationMappingFileCommand(projectKind,languageClient,weaveToolingService,virtualFileSystem),
      new InstallBatCommand(clientLogger, projectKind),
      new RunWeaveCommand(virtualFileSystem, projectVirtualFileSystem, project, projectKind, clientLogger, jobManagerService, languageClient, dataWeaveTestService),
      new ReloadDependenciesCommand(projectKind),
      new CreateUnitTest(weaveToolingService, languageClient, projectKind, project)
    ) ++ projectKind.customCommands()
    projectDependentCommands = Some(commands.map(prefixCommand(commandPrefix, _)))
  }

  var projectDependentCommands: Option[Seq[WeaveCommand]] = None

  lazy val projectIndependentCommands: Seq[WeaveCommand] = Seq(
    new CreateScenarioCommand(languageClient, scenariosManager),
    new DeleteScenarioCommand(scenariosManager),
    new SetActiveScenarioCommand(scenariosManager),
    new DeleteInputSampleCommand(scenariosManager),
    new CreateInputSampleCommand(languageClient, weaveToolingService, weaveAgentService, dataFormatsService, scenariosManager, project),
    new CreateMappingTestCommand(languageClient, scenariosManager, virtualFileSystem, previewService),
    new DeleteOutputSampleCommand(scenariosManager),
    new EnablePreviewModeCommand(previewService, virtualFileSystem),
    new RunPreviewCommand(previewService, virtualFileSystem, languageClient),
    new LaunchWeaveCommand(languageClient),
    new LaunchWeaveTestCommand(languageClient, virtualFileSystem),
    new QuickFixCommand(weaveToolingService),
    new InsertDocumentationCommand(weaveToolingService),
    new InsertWeaveTypeCommand(weaveToolingService, project),
    new ExtractVariableCommand(weaveToolingService),
    new ExtractConstantCommand(weaveToolingService),
    new ExtractFunctionCommand(weaveToolingService, languageClient),
    new RestartAgentCommand(weaveAgentService)
  ).map(prefixCommand(commandPrefix, _))

  def prefixCommand(prefix: Option[String], weaveCommand: WeaveCommand): WeaveCommand = {
    prefix
      .map(PrefixedWeaveCommand(_, weaveCommand))
      .getOrElse(weaveCommand)
  }

  def commands: Seq[WeaveCommand] =
    projectIndependentCommands ++ projectDependentCommands.getOrElse(Seq())

  def commandBy(commandId: String): Option[WeaveCommand] = {
    commands.find(_.commandId() == commandId)
  }

  def prefixCommandBaseId(baseCommandId: String): String = {
    commandPrefix.getOrElse("") ++ baseCommandId
  }

  def commandIds(): Seq[String] = {
    commands.map(_.commandId())
  }

}
