package org.mule.weave.lsp.services


import org.eclipse.lsp4j.Diagnostic
import org.eclipse.lsp4j.DiagnosticSeverity
import org.eclipse.lsp4j.PublishDiagnosticsParams
import org.eclipse.lsp4j.services.LanguageClient
import org.mule.weave.lsp.indexer.events.IndexingFinishedEvent
import org.mule.weave.lsp.indexer.events.IndexingType
import org.mule.weave.lsp.indexer.events.IndexingType.IndexingType
import org.mule.weave.lsp.indexer.events.OnIndexingFinished
import org.mule.weave.lsp.project.Project
import org.mule.weave.lsp.project.ProjectKind
import org.mule.weave.lsp.project.Settings
import org.mule.weave.lsp.project.components.ContextMetadata
import org.mule.weave.lsp.project.events.OnProjectStarted
import org.mule.weave.lsp.project.events.OnSettingsChanged
import org.mule.weave.lsp.project.events.ProjectStartedEvent
import org.mule.weave.lsp.project.events.SettingsChangedEvent
import org.mule.weave.lsp.utils.EventBus
import org.mule.weave.lsp.utils.LSPConverters.toDiagnostic
import org.mule.weave.lsp.utils.LSPConverters.toDiagnosticKind
import org.mule.weave.lsp.utils.WeaveTypeUtils
import org.mule.weave.lsp.utils.URLUtils
import org.mule.weave.lsp.vfs.events.OnProjectVirtualFileChangedEvent
import org.mule.weave.lsp.vfs.events.OnProjectVirtualFileCreatedEvent
import org.mule.weave.lsp.vfs.events.OnProjectVirtualFileDeletedEvent
import org.mule.weave.lsp.vfs.events.ProjectVirtualFileChangedEvent
import org.mule.weave.lsp.vfs.events.ProjectVirtualFileCreatedEvent
import org.mule.weave.lsp.vfs.events.ProjectVirtualFileDeletedEvent
import org.mule.weave.v2.editor.ImplicitInput
import org.mule.weave.v2.editor.MappingInput
import org.mule.weave.v2.editor.QuickFix
import org.mule.weave.v2.editor.ValidationMessage
import org.mule.weave.v2.editor.ValidationMessages
import org.mule.weave.v2.editor.VirtualFile
import org.mule.weave.v2.editor.VirtualFileSystem
import org.mule.weave.v2.editor.WeaveDocumentToolingService
import org.mule.weave.v2.editor.WeaveToolingService
import org.mule.weave.v2.parser.ast.variables.NameIdentifier
import org.mule.weave.v2.ts.WeaveType
import org.mule.weave.v2.versioncheck.SVersion

import java.util
import java.util.concurrent.CompletableFuture
import java.util.concurrent.Executor
import java.util.logging.Level
import java.util.logging.Logger

class DataWeaveToolingService(val project: Project, languageClient: LanguageClient, val vfs: VirtualFileSystem, documentServiceFactory: () => WeaveToolingService, executor: Executor) extends ToolingService {


  private val logger: Logger = Logger.getLogger(getClass.getName)
  protected var projectKind: ProjectKind = _
  private lazy val _documentService: WeaveToolingService = documentServiceFactory()

  @volatile
  protected var indexed: Boolean = false

  override def init(projectKind: ProjectKind, eventBus: EventBus): Unit = {
    this.projectKind = projectKind
    eventBus.register(SettingsChangedEvent.SETTINGS_CHANGED, new OnSettingsChanged {
      override def onSettingsChanged(modifiedSettingsName: Array[String]): Unit = {
        if (modifiedSettingsName.contains(Settings.LANGUAGE_LEVEL_PROP_NAME)) {
          validateAllEditors("settingsChanged")
        }
      }
    })

    eventBus.register(IndexingFinishedEvent.INDEXING_FINISHED, new OnIndexingFinished() {
      override def onIndexingFinished(indexingType: IndexingType): Unit = {
        if (IndexingType.Dependencies.equals(indexingType)) {
          indexed = true
          validateAllEditors("indexingFinishes")
        }
      }
    })

    eventBus.register(ProjectStartedEvent.PROJECT_STARTED, new OnProjectStarted {
      override def onProjectStarted(project: Project): Unit = {
        validateAllEditors("projectStarted")
      }
    })
    eventBus.register(ProjectVirtualFileCreatedEvent.VIRTUAL_FILE_CREATED, new OnProjectVirtualFileCreatedEvent {
      override def onVirtualFileCreated(vf: VirtualFile): Unit = {
        if (projectKind.isDWFile(vf.url())) {
          validateFile(vf, "onCreated")
        } else {
          // Validate file if a scenario is modified
          findCorrespondingDwFile(vf.url()).foreach(dwVirtualFile => validateFile(dwVirtualFile, "onScenarioChanged"))
        }
      }
    })
    eventBus.register(ProjectVirtualFileChangedEvent.VIRTUAL_FILE_CHANGED, new OnProjectVirtualFileChangedEvent {
      override def onVirtualFileChanged(vf: VirtualFile): Unit = {
        if (projectKind.isDWFile(vf.url())) {
          validateFile(vf, "onChanged")
        } else {
          // Validate file if a scenario is modified
          findCorrespondingDwFile(vf.url()).foreach(dwVirtualFile => validateFile(dwVirtualFile, "onScenarioChanged"))
        }
      }
    })
    eventBus.register(ProjectVirtualFileDeletedEvent.VIRTUAL_FILE_DELETED, new OnProjectVirtualFileDeletedEvent {
      override def onVirtualFileDeleted(vf: VirtualFile): Unit = {
        if (projectKind.isDWFile(vf.url())) {
          validateDependencies(vf, "onDeleted")
        } else {
          // Validate file if a scenario is modified
          findCorrespondingDwFile(vf.url()).foreach(dwVirtualFile => validateFile(dwVirtualFile, "onScenarioChanged"))
        }
      }
    })
  }

  private def findCorrespondingDwFile(str: String): Option[VirtualFile] = {
    val sampleDataManager = projectKind.sampleDataManager()
    documentService().openEditors().find((oe) => {
      URLUtils.isChildOfAny(str, sampleDataManager.listScenarios(oe.file.getNameIdentifier).map(scenario => scenario.file))
    }).map(weaveDocToolingService => weaveDocToolingService.file)
  }

  private def validateAllEditors(reason: String): Unit = {
    val services = documentService().openEditors().map(openEditor => openEditor.file.url());
    documentService().invalidateAll()
    services.foreach((url) => {
      triggerValidation(url, reason)
    })
  }


  private def validateDependencies(vf: VirtualFile, reason: String): Unit = {
    val fileLogicalName: NameIdentifier = vf.getNameIdentifier
    val dependants: Seq[NameIdentifier] = documentService().dependantsOf(fileLogicalName)
    logger.log(Level.INFO, s"[DWProject] Validate dependants of $fileLogicalName: ${dependants.mkString("[", ",", "]")}")
    dependants.foreach((ni) => {
      vf.fs().asResourceResolver.resolve(ni) match {
        case Some(resource) => {
          triggerValidation(resource.url(), "dependantChanged ->" + reason + s" ${vf.url()}")
        }
        case None => {
          logger.log(Level.WARNING, "No resource found for file " + vf.url())
        }
      }
    })
  }

  def loadType(typeString: String): Option[WeaveType] = {
    documentService().loadType(typeString)
  }

  private def documentService(): WeaveToolingService = {
    _documentService
  }

  /**
    * Returns the input directive that has the same name as the one specified in the specified document.
    *
    * @param uri       The uri of the document to search in
    * @param inputName The name of the input
    * @return The input directive if any
    */
  def inputOf(uri: String, inputName: String): Option[MappingInput] = {
    val documentToolingService = openDocument(uri)
    documentToolingService.inputOf(inputName)
  }
  
  def contextMetadataFor(uri: String, withExpectedOutput: Boolean = true): Option[ContextMetadata] = {
    projectKind.metadataProvider().flatMap(provider => {
      val virtualFile = vfs.file(uri)
      if (virtualFile != null) {
        Some(provider.metadataFor(virtualFile, withExpectedOutput))
      } else {
        None
      }
    })
  }

  def openDocument(uri: String, withExpectedOutput: Boolean = true): WeaveDocumentToolingService = {
    val maybeContext: Option[ContextMetadata] = contextMetadataFor(uri, withExpectedOutput)
    if (maybeContext.isDefined) {
      openDocument(uri, maybeContext.get)
    } else {
      openDocumentWithoutContext(uri)
    }
  }

  def openDocument(uri: String, maybeContext: Option[ContextMetadata], withExpectedOutput: Boolean): WeaveDocumentToolingService = {
    if (maybeContext.isDefined) {
      openDocument(uri, maybeContext.get)
    } else {
      openDocument(uri, withExpectedOutput)
    }
  }

  def openDocumentWithoutContext(uri: String): WeaveDocumentToolingService = {
    _documentService.open(uri, ImplicitInput(), None)
  }

  private def openDocument(uri: String, contextMetadata: ContextMetadata): WeaveDocumentToolingService = {
    _documentService.open(uri, WeaveTypeUtils.toImplicitInput(contextMetadata.input.metadata), contextMetadata.output)
  }

  def closeDocument(uri: String): Unit = {
    documentService().close(uri)
  }

  def withLanguageLevel(dwLanguageLevel: String): WeaveToolingService = {
    _documentService.updateLanguageLevel(SVersion.fromString(dwLanguageLevel))
  }


  def validateFile(vf: VirtualFile, reason: String): Unit = {
    triggerValidation(vf.url(), reason, () => validateDependencies(vf, reason))
  }

  /**
    * Triggers the validation of the specified document.
    *
    * @param documentUri        The URI to be validated
    * @param onValidationFinish A Callback that is called when the validation finishes
    */
  def triggerValidation(documentUri: String, reason: String, onValidationFinish: () => Unit = () => {}): Unit = {
    logger.log(Level.INFO, "TriggerValidation of: " + documentUri + " reason " + reason)
    CompletableFuture.runAsync(() => {
      logger.log(Level.INFO, s"[${Thread.currentThread().getName}][${this.getClass.getName}] Init TriggerValidation of: " + documentUri + " reason " + reason)
      withLanguageLevel(projectKind.dependencyManager().languageLevel())
      val messages: ValidationMessages = validate(documentUri)
      val diagnostics = toDiagnostics(messages)
      logger.log(Level.INFO, s"[${Thread.currentThread().getName}][${this.getClass.getName}] TriggerValidation finished: " + documentUri + " reason " + reason + s". Diagnostics: [${diagnostics}]")
      languageClient().publishDiagnostics(new PublishDiagnosticsParams(documentUri, diagnostics))
      onValidationFinish()

    }, executor)
  }

  def toDiagnostics(messages: ValidationMessages): util.List[Diagnostic] = {
    val diagnostics = new util.ArrayList[Diagnostic]
    messages.errorMessage.foreach(message => {
      diagnostics.add(toDiagnostic(message, DiagnosticSeverity.Error))
    })

    messages.warningMessage.foreach((message) => {
      diagnostics.add(toDiagnostic(message, DiagnosticSeverity.Warning))
    })
    diagnostics
  }

  def quickFixesFor(documentUri: String, startOffset: Int, endOffset: Int, kind: String, severity: String, maybeContext: Option[ContextMetadata] = None): Array[QuickFix] = {
    val messages: ValidationMessages = validate(documentUri, maybeContext)
    val messageFound: Option[ValidationMessage] = if (severity == DiagnosticSeverity.Error.name()) {
      messages.errorMessage.find((m) => {
        matchesMessage(m, kind, startOffset, endOffset)
      })
    } else {
      messages.warningMessage.find((m) => {
        matchesMessage(m, kind, startOffset, endOffset)
      })
    }
    messageFound.map(_.quickFix).getOrElse(Array.empty)
  }

  private def matchesMessage(m: ValidationMessage, kind: String, startOffset: Int, endOffset: Int): Boolean = {
    m.location.startPosition.index == startOffset && m.location.endPosition.index == endOffset && toDiagnosticKind(m) == kind
  }

  /**
    * Executes Weave Validation into the corresponding type level
    *
    * @param documentUri The URI to be validates
    * @return The Validation Messages
    */
  def validate(documentUri: String, maybeContext: Option[ContextMetadata] = None): ValidationMessages = {
    val documentToolingService = if (maybeContext.isDefined) {
      val context = maybeContext.get
      openDocument(documentUri, context)
    } else {
      openDocument(documentUri, withExpectedOutput = false)
    }
    val messages: ValidationMessages =
      if (indexed && Settings.isTypeLevel(project.settings)) {
        documentToolingService.typeCheck()
      } else if (indexed && Settings.isScopeLevel(project.settings)) {
        documentToolingService.scopeCheck()
      } else {
        documentToolingService.parseCheck()
      }
    messages
  }

  def languageClient(): LanguageClient = {
    languageClient
  }

  def getLogger(): Logger = this.logger

}
