package org.mule.weave.lsp.project

import org.eclipse.lsp4j.ResourceOperation
import org.eclipse.lsp4j.TextDocumentEdit
import org.eclipse.lsp4j.jsonrpc.messages.Either
import org.mule.weave.lsp.actions.CodeActionProvider
import org.mule.weave.lsp.commands.WeaveCommand
import org.mule.weave.lsp.project.components.BuildManager
import org.mule.weave.lsp.project.components.DefaultProjectFolderFactory
import org.mule.weave.lsp.project.components.MetadataProvider
import org.mule.weave.lsp.project.components.ProjectDependencyManager
import org.mule.weave.lsp.project.components.ProjectFolderFactory
import org.mule.weave.lsp.project.components.ProjectStructure
import org.mule.weave.lsp.project.components.SampleDataManager
import org.mule.weave.lsp.project.impl.maven.MavenProjectKind
import org.mule.weave.lsp.utils.URLUtils.toURI

import java.io.File

/**
  * A Project Kind is the trait that allows us to support multiple kind of projects with different:
  *  - builds
  *  - dependency management
  *  - folder structure
  */
trait ProjectKind {

  /**
    * The name of the kind i.e maven, bat, simple ...
    *
    * @return
    */
  def name(): String

  /**
    * Setups the project kind. Download any tool required for this kind of project
    */
  def start(): Unit = {}

  /**
    * Returns the Project structure with all the modules and it source folders
    *
    * @return
    */
  def structure(): ProjectStructure

  /**
    * Returns whether a uri is a file of the given kind project
    *
    * @param uri the uri
    * @return
    */
  def isProjectFile(uri: String): Boolean = {
    ProjectStructure.isAProjectFile(uri, structure())
  }

  /**
    * The Dependency Manager handles all the dependencies for this kind of projects
    *
    * @return
    */
  def dependencyManager(): ProjectDependencyManager

  /**
    * Handles the build, deploy of this project
    *
    * @return
    */
  def buildManager(): BuildManager

  /**
    * Handles Scenario Provider for sample data
    *
    * @return The Scenarios For sample data
    */
  def sampleDataManager(): SampleDataManager

  /**
    * Handles the metadata for the files.
    *
    * @return The Metadata Provider if this Project can provide any
    */
  def metadataProvider(): Option[MetadataProvider] = None

  /**
    * Returns any additional change that needs to be made when a new file is created
    *
    * @param folder the folder
    * @param name the file name
    * @return
    */
  def newFile(folder: File, name: String): Array[Either[TextDocumentEdit, ResourceOperation]] = Array.empty

  /**
    * Returns the folder factory for the given project
    *
    * @return
    */
  def projectFolderFactory(): ProjectFolderFactory = DefaultProjectFolderFactory

  /**
    * Returns whether a schema uri support edits
    * 
    * @param uri the uri
    * @return Return true if the schema uri support edits
    */
  def isSupportedEditableScheme(uri: String): Boolean = {
    toURI(uri)
      .exists(uri => {
        val scheme: String = uri.getScheme
        scheme == "file" || scheme == "untitled"
      })
  }

  /**
    * Returns whether an uri belongs to a DW file.
    * By default uris endings with ".dwl" is considered a DW file.
    * 
    * @param uri the uri
    * @return Return true if an uri belongs to a DW file.
    */
  def isDWFile(uri: String): Boolean = {
    uri.endsWith(".dwl")
  }

  /**
    * Get a custom set of CodeActionProvider according to the project kind
    * @return A custom set of CodeActionProvider according to the project kind
    */
  def customCodeActions(): Array[CodeActionProvider] = Array.empty

  /**
   * Returns a custom set of WeaveCommands according to the project kind
   */
  def customCommands(): Array[WeaveCommand] = Array.empty

  /**
    * Get the DataWeave version for the current project kind. For maven projects
    * it's obtained first from the POM, otherwise it defaults to the dependency manager language level.
    *
    * @return The current dataweave version in major.minor format (2.4, 2.5).
    */
  def getWeaveVersion(): String = {
    val version = dependencyManager().languageLevel().split("\\.")
    s"${version(0)}.${version(1)}"
  }
}
