package org.mule.weave.lsp.project

import org.mule.weave.lsp.agent.WeaveAgentService
import org.mule.weave.lsp.commands.CommandProvider
import org.mule.weave.lsp.extension.client.WeaveLanguageClient
import org.mule.weave.lsp.jobs.JobManagerService
import org.mule.weave.lsp.project.impl.simple.SimpleProjectKind
import org.mule.weave.lsp.services.ClientLogger
import org.mule.weave.lsp.services.DataWeaveToolingService
import org.mule.weave.lsp.services.PreviewService
import org.mule.weave.lsp.services.WeaveScenarioManagerService
import org.mule.weave.lsp.utils.EventBus

import java.util
import java.util.ServiceLoader
import scala.collection.JavaConverters._

/**
  * Detects the Project Kind
  */
trait ProjectKindDetector {

  /**
    * Returns true if this Kind detector handles the specific project
    *
    * @param context The project that we want to check, and all the context needed to check
    * @return true if it is supported
    */
  def supports(context: ProjectKindContext): Boolean


  /**
    * Returns the project kind that was detected
    *
    * @param context The project to be used, and all the context needed to create the kind
    * @return The ProjectKind
    */
  def createKind(context: ProjectKindContext): ProjectKind

}

case class ProjectKindContext(project: Project,
                              eventBus: EventBus,
                              clientLogger: ClientLogger,
                              weaveAgentService: WeaveAgentService,
                              weaveLanguageClient: WeaveLanguageClient,
                              weaveScenarioManagerService: WeaveScenarioManagerService,
                              jobManagerService: JobManagerService,
                              toolingService: DataWeaveToolingService,
                              previewService: PreviewService,
                              commandProvider: CommandProvider)

class ProjectKindDetectorManager {

  def detectProjectKind(project: Project,
                        eventBus: EventBus,
                        clientLogger: ClientLogger,
                        weaveAgentService: WeaveAgentService,
                        weaveLanguageClient: WeaveLanguageClient,
                        weaveScenarioManagerService: WeaveScenarioManagerService,
                        jobManagerService: JobManagerService,
                        toolingService: DataWeaveToolingService, 
                        previewService: PreviewService,
                        commandProvider: CommandProvider): ProjectKind = {
    if (project.hasHome) {
      val context = ProjectKindContext(project,
        eventBus,
        clientLogger,
        weaveAgentService,
        weaveLanguageClient,
        weaveScenarioManagerService,
        jobManagerService,
        toolingService,
        previewService,
        commandProvider)
      val kindDetectors: Seq[ProjectKindDetector] = projectKindDetectors().asScala
      kindDetectors
        .find(_.supports(context))
        .map(_.createKind(context))
        .getOrElse(new SimpleProjectKind(project, clientLogger, eventBus, weaveAgentService, weaveScenarioManagerService))
    } else {
      new SimpleProjectKind(project, clientLogger, eventBus, weaveAgentService, weaveScenarioManagerService)
    }
  }

  def projectKindDetectors(): util.List[ProjectKindDetector] = {
    val detectors: util.Iterator[ProjectKindDetector] = ServiceLoader.load(classOf[ProjectKindDetector], classOf[ProjectKindDetector].getClassLoader).iterator()
    if (!detectors.hasNext) {
      throw new RuntimeException("No Project Kind Detectors Were Defined")
    }
    detectors.asScala.toSeq.asJava
  }
}