package org.mule.weave.lsp.project.impl.maven

import org.eclipse.lsp4j.FileChangeType
import org.mule.weave.lsp.jobs.JobManagerService
import org.mule.weave.lsp.jobs.Status
import org.mule.weave.lsp.project.Project
import org.mule.weave.lsp.project.components.DependencyArtifact
import org.mule.weave.lsp.project.components.ProjectDependencyManager
import org.mule.weave.lsp.project.events.DependencyArtifactRemovedEvent
import org.mule.weave.lsp.project.events.DependencyArtifactResolvedEvent
import org.mule.weave.lsp.services.ClientLogger
import org.mule.weave.lsp.services.events.FileChangedEvent
import org.mule.weave.lsp.services.events.OnFileChanged
import org.mule.weave.lsp.utils.EventBus
import org.mule.weave.lsp.utils.JavaLoggerForwarder.interceptLog
import org.mule.weave.lsp.utils.URLUtils

import java.io.File

import scala.collection.JavaConverters._

class MavenProjectDependencyManager(project: Project, pomFile: File, eventBus: EventBus, loggerService: ClientLogger, jobManagerService: JobManagerService) extends ProjectDependencyManager {
  var dependenciesArray: Array[DependencyArtifact] = Array.empty
  var languageVersion: String = project.settings.wlangVersion.value()

  override def start(): Unit = {
    reloadArtifacts()
    eventBus.register(FileChangedEvent.FILE_CHANGED_EVENT, new OnFileChanged {
      override def onFileChanged(uri: String, changeType: FileChangeType): Unit = {
        changeType match {
          case FileChangeType.Changed =>
            val maybeFile = URLUtils.toFile(uri)
            if (maybeFile.exists(_.equals(pomFile))) {
              reload()
            }
          case _ =>
        }
      }
    })
  }

  override def reload(): Unit = {
    jobManagerService.schedule((_: Status) => {
      reloadArtifacts()
    }, "Loading Maven Dependencies", "Loading Maven Dependencies.")
  }

  private def reloadArtifacts(): Unit = {
    try {
      loggerService.logDebug("Loading artifacts from : " + pomFile.getPath)
      interceptLog(loggerService) {
        val dependencies = MavenDependencyResolver(pomFile).resolve().asScala
        
        if (dependencies.nonEmpty) {
          eventBus.fire(new DependencyArtifactRemovedEvent(dependenciesArray))
          languageVersion = dependencies
            .find(d => d.getCoordinate.getArtifactId.equals("wlang"))
            .map(d => d.getCoordinate.getVersion)
            .getOrElse(project.settings.wlangVersion.value())

          dependenciesArray = dependencies
            .map(a => {
              DependencyArtifact(a.getCoordinate.toCanonicalForm, a.asFile())
            }).toArray
          eventBus.fire(new DependencyArtifactResolvedEvent(dependenciesArray))
          loggerService.logDebug("All dependencies were loaded successfully.")
        } else {
          loggerService.logDebug("No dependency was detected.")
          eventBus.fire(new DependencyArtifactRemovedEvent(dependenciesArray))
          eventBus.fire(new DependencyArtifactResolvedEvent(Array()))
        }
      }
    } catch {
      case e: Exception => loggerService.logError(s"Exception while resolving dependencies from ${pomFile.getAbsolutePath}", e)
    }
  }

  override def languageLevel(): String = {
    languageVersion
  }

  /**
    * Returns the list of dependencies
    *
    * @return
    */
  def dependencies(): Array[DependencyArtifact] = {
    dependenciesArray
  }
}
