package org.mule.weave.lsp.vfs.resource

import com.google.common.cache.Cache
import com.google.common.cache.CacheBuilder
import org.apache.commons.io.FileUtils
import org.eclipse.lsp4j.FileChangeType
import org.mule.weave.lsp.agent.WeaveAgentService
import org.mule.weave.lsp.project.components.DependencyArtifact
import org.mule.weave.lsp.project.events.DependencyArtifactResolvedEvent
import org.mule.weave.lsp.project.events.OnDependencyArtifactResolved
import org.mule.weave.lsp.services.events.AgentStartedEvent
import org.mule.weave.lsp.services.events.FileChangedEvent
import org.mule.weave.lsp.services.events.OnAgentStarted
import org.mule.weave.lsp.services.events.OnFileChanged
import org.mule.weave.lsp.utils.EventBus
import org.mule.weave.v2.parser.ast.variables.NameIdentifier
import org.mule.weave.v2.sdk.WeaveResource
import org.mule.weave.v2.sdk.WeaveResourceResolver
import org.mule.weave.v2.utils.WeaveFile

import java.io.File
import java.nio.charset.StandardCharsets
import java.nio.file.Files
import java.util.concurrent.TimeUnit
import scala.collection.mutable

class CustomLoaderResourceResolver(weaveAgentService: () => WeaveAgentService, val loader: String, val fileExtension: String, eventBus: EventBus) extends WeaveResourceResolver {

  private val resourceCache: Cache[String, Option[WeaveResource]] = CacheBuilder.newBuilder().build[String, Option[WeaveResource]]()

  private val uriToNameIdentifier = mutable.HashMap[String, NameIdentifier]()

  private val originalFileUriToStubUri = mutable.HashMap[String, String]()

  eventBus.register(FileChangedEvent.FILE_CHANGED_EVENT, new OnFileChanged {
    override def onFileChanged(uri: String, changeType: FileChangeType): Unit = {
      if (uri.endsWith("." + fileExtension)) {
        //TODO we can do better but for know this may be good enough
        resourceCache.invalidateAll()
      }
    }
  })

  lazy val stubsFolder: File = {
    Files.createTempDirectory(loader).toFile
  }

  /**
    * Returns the folder where all the stubs are going to be persisted
    *
    * @return
    */
  def stubsModulesFolder: File = {
    stubsFolder
  }

  def nameIdentifierOfStub(uri: String): Option[NameIdentifier] = {
    uriToNameIdentifier.get(uri)
  }

  def nameIdentifierOfOriginalFile(uri: String): Option[NameIdentifier] = {
    originalFileUriToStubUri.get(uri).flatMap(nameIdentifierOfStub)
  }

  eventBus.register(DependencyArtifactResolvedEvent.ARTIFACT_RESOLVED, new OnDependencyArtifactResolved {
    override def onArtifactsResolved(artifacts: Array[DependencyArtifact]): Unit = {
      resourceCache.invalidateAll()
    }
  })

  eventBus.register(AgentStartedEvent.AGENT_STARTED, new OnAgentStarted {
    override def onAgentStarted(): Unit = {
      resourceCache.invalidateAll()
    }
  })

  override def resolve(name: NameIdentifier): Option[WeaveResource] = {
    if (name.loader.exists(_.equals(loader))) {
      resourceCache.get(name.toString(), () => {
        val maybeResource: Option[WeaveResource] = weaveAgentService()
          .resolveModule(name.toString(), loader)
          .get(30L, TimeUnit.SECONDS)
        maybeResource.map((wr) => {
          val file = new File(stubsModulesFolder, name.toString().replace(NameIdentifier.SEPARATOR, File.separator) + WeaveFile.fileExtension)
          file.getParentFile.mkdirs()
          val tmpFilePath = file.toPath
          val stubUri = tmpFilePath.toUri.toString
          uriToNameIdentifier.put(stubUri, name)
          originalFileUriToStubUri.put(wr.url(), stubUri)
          FileUtils.writeStringToFile(tmpFilePath.toFile, wr.content(), StandardCharsets.UTF_8)
          WeaveResource(tmpFilePath.toFile.toURI.toString, wr.content())
        })
      })
    } else {
      None
    }

  }
}
