package org.mule.weave.lsp.commands

import org.eclipse.lsp4j.ApplyWorkspaceEditParams
import org.eclipse.lsp4j.Command
import org.eclipse.lsp4j.CreateFile
import org.eclipse.lsp4j.ExecuteCommandParams
import org.eclipse.lsp4j.MessageParams
import org.eclipse.lsp4j.MessageType
import org.eclipse.lsp4j.Position
import org.eclipse.lsp4j.Range
import org.eclipse.lsp4j.ResourceOperation
import org.eclipse.lsp4j.TextDocumentEdit
import org.eclipse.lsp4j.TextEdit
import org.eclipse.lsp4j.VersionedTextDocumentIdentifier
import org.eclipse.lsp4j.WorkspaceEdit
import org.eclipse.lsp4j.jsonrpc.messages.Either
import org.mule.weave.lsp.extension.client.OpenTextDocumentParams
import org.mule.weave.lsp.extension.client.WeaveLanguageClient
import org.mule.weave.lsp.i18n.UserMessages
import org.mule.weave.lsp.i18n.UserMessages.invalidTestFileSyntax
import org.mule.weave.lsp.project.Project
import org.mule.weave.lsp.project.ProjectKind
import org.mule.weave.lsp.project.components.ProjectStructure
import org.mule.weave.lsp.services.DataWeaveToolingService
import org.mule.weave.lsp.utils.URLUtils.toLSPUrl
import org.mule.weave.lsp.utils.WeaveDirectoryUtils
import org.mule.weave.v2.editor.WeaveDocumentToolingService
import org.mule.weave.v2.editor.WeaveUnitTestAddition
import org.mule.weave.v2.parser.ast.AstNode
import org.mule.weave.v2.parser.location.WeaveLocation

import java.io.File
import java.util
import java.util
import scala.io.Source

class CreateUnitTest(validationService: DataWeaveToolingService, weaveLanguageClient: WeaveLanguageClient, projectKind: ProjectKind, project: Project) extends WeaveCommand {
  override def commandId(): String = Commands.DW_CREATE_UNIT_TEST

  override def execute(params: ExecuteCommandParams): AnyRef = {
    val args: util.List[AnyRef] = params.getArguments

    val uri: String = Commands.argAsString(args, 0)
    val startOffset: Int = Commands.argAsInt(args, 1)
    val endOffset: Int = Commands.argAsInt(args, 2)

    val documentToolingService: WeaveDocumentToolingService = validationService.openDocument(uri)
    val testPath: String = documentToolingService.getTestPathFromDefinition()
    val files = ProjectStructure.testsSourceFolders(projectKind.structure())
    val weaveTestFolder: File =
      files.find((f) => f.getName == WeaveDirectoryUtils.DWTest_FOLDER)
        .getOrElse(projectKind.projectFolderFactory().createTestSourceFolder(project))
    val testFile: File = new File(weaveTestFolder, testPath)
    val testFileURL: String = toLSPUrl(testFile)
    doCreateTest(testFile, testFileURL, documentToolingService, startOffset, endOffset)
    null
  }

  private def doCreateTest(testFile: File, testFileURL: String, documentToolingService: WeaveDocumentToolingService, startOffset: Int, endOffset: Int): Unit = {
    if (isBlankFile(testFile)) {
      addTestToBlankFile(testFile, testFileURL, documentToolingService, startOffset, endOffset)
    } else {
      addTestToExistingTestCode(testFile, testFileURL, documentToolingService, startOffset, endOffset)
    }
  }

  private def addTestToBlankFile(testFile: File, testFileURL: String, documentToolingService: WeaveDocumentToolingService, startOffset: Int, endOffset: Int): Unit = {
    val maybeTest: Option[String] = documentToolingService.createUnitTestFromDefinition(startOffset, endOffset)
    maybeTest.foreach((test) => {
      val insertText = insertTextEdit(testFileURL, test, 0)
      val createFile = Either.forRight[TextDocumentEdit, ResourceOperation](new CreateFile(testFileURL))
      val edits = if (!testFile.exists()) Seq(createFile, insertText) else Seq(insertText)
      applyEdits(edits: _*)
      weaveLanguageClient.openTextDocument(OpenTextDocumentParams(testFileURL))
    })
  }

  type Edit = Either[TextDocumentEdit, ResourceOperation]

  private def addTestToExistingTestCode(testFile: File, testFileURL: String, documentToolingService: WeaveDocumentToolingService, startOffset: Int, endOffset: Int): Unit = {
    val testToolingService: WeaveDocumentToolingService = validationService.openDocument(testFileURL)
    val maybeAddition = getTestAddition(testToolingService, documentToolingService, startOffset, endOffset)
    if (maybeAddition.isEmpty) {
      unableToRunTest(testFile)
      return
    }
    val testAddition: WeaveUnitTestAddition = maybeAddition.get
    val testStr = testAddition.test
    val line: Int = testAddition.replacementLine - 1 //lsp uses 0-based lines

    val edits = collection.mutable.ArrayBuffer[Edit]()
    if (testAddition.needsCommaBefore) {
      val commaPosition = testAddition.commaPosition
      edits.+=(insertTextEdit(testFileURL, ",", commaPosition.line - 1, commaPosition.column - 1))
    }
    edits.+=(insertTextEdit(testFileURL, testStr, line))
    applyEdits(edits: _*)
    weaveLanguageClient.openTextDocument(OpenTextDocumentParams(
      testFileURL,
      new Range(new Position(line, 0), new Position(line + testStr.count(_ == '\n'), 0)),
    ))
  }

  private def getTestAddition(testToolingService: WeaveDocumentToolingService, documentToolingService: WeaveDocumentToolingService, startOffset: Int, endOffset: Int): Option[WeaveUnitTestAddition] = {
    val maybeAst = testToolingService.ast()
    maybeAst.flatMap(ast => {
      documentToolingService.addUnitTestFromDefinition(startOffset, endOffset, ast)
    })
  }

  private def insertTextEdit(testFileURL: String, test: String, line: Int, column: Int = 0): Edit = {
    val position = new Position(line, column)
    val textEdit = new TextEdit(new Range(position, position), test)
    val textDocumentEdit = new TextDocumentEdit(new VersionedTextDocumentIdentifier(testFileURL, 0), util.Arrays.asList(textEdit))
    Either.forLeft[TextDocumentEdit, ResourceOperation](textDocumentEdit)
  }

  private def applyEdits(edits: Either[TextDocumentEdit, ResourceOperation]*): Unit = {
    val editList = util.Arrays.asList(edits: _*)
    weaveLanguageClient.applyEdit(new ApplyWorkspaceEditParams(new WorkspaceEdit(editList))).get()
  }

  private def isBlankFile(testFile: File) = {
    if (!testFile.exists()) {
      true
    } else {
      val source = Source.fromFile(testFile, "UTF-8")
      try {
        source.mkString.trim.isEmpty
      } finally {
        source.close()
      }
    }
  }

  private def unableToRunTest(testFile: File): Unit = {
    weaveLanguageClient.showMessage(new MessageParams(MessageType.Error, invalidTestFileSyntax(testFile)))
  }

  override def name(): String = "Creates Unit."

  override def description(params: ExecuteCommandParams): String = "Creating Unit Test."
}

object CreateUnitTest {
  def createCommand(uri: String, astNode: AstNode, commandProvider: CommandProvider): Command = {
    val nodeLocation: WeaveLocation = astNode.location()
    new Command("Add Unit Test",
      commandProvider.prefixCommandBaseId(Commands.DW_CREATE_UNIT_TEST),
      util.Arrays.asList(
        uri,
        nodeLocation.startPosition.index: java.lang.Integer,
        nodeLocation.endPosition.index: java.lang.Integer,
      )
    )
  }

}
