package org.mule.weave.lsp.commands

import org.eclipse.lsp4j.ExecuteCommandParams
import org.mule.weave.lsp.bat.BatProjectHelper
import org.mule.weave.lsp.bat.BatUserMessages
import org.mule.weave.lsp.extension.client.WeaveLanguageClient
import org.mule.weave.lsp.project.ProjectKind
import org.mule.weave.lsp.utils.Messages
import org.mule.weave.lsp.utils.WeaveASTQueryUtils
import org.mule.weave.v2.editor.VirtualFile

/**
  * Trait that represent any weave command
  */
trait WeaveCommand {

  /**
    * The Id of the underlying command
    *
    * @return
    */
  def commandId(): String

  /**
    * Returns the name of the command
    */
  def name(): String

  /**
    * Returns a description of the current command is doing. This message is going to be presented to the user as information of
    * what the command is running
    *
    * @return A short description of what this command is doing
    */
  def description(params: ExecuteCommandParams): String

  /**
    * Executes the given command with the specified parameters
    *
    * @param params The command parameters
    * @return
    */
  def execute(params: ExecuteCommandParams): AnyRef

  /**
    * Returns whether this command should run on the given context.
    *
    * @return
    */
  def enabled(vf: VirtualFile): Boolean = true

}

abstract class BatCommand(projectKind: ProjectKind) extends WeaveCommand {

  override def enabled(vf: VirtualFile): Boolean = {
    projectKind.name().equals(WeaveASTQueryUtils.BAT)
  }


  protected def installCLIIfNeeded(batProjectManager: BatProjectHelper, languageClient: WeaveLanguageClient): Boolean = {
    if (!batProjectManager.isBatInstalled) {
      val value = languageClient.showMessageRequest(BatUserMessages.installBatCLI())
      value.get() match {
        case msg if msg == Messages.no =>
          false
        case msg if msg == Messages.yes =>
          batProjectManager.downloadAndInstall()
        case _ => {
          false
        }
      }
    } else {
      true
    }

  }

}








