package org.mule.weave.lsp.vfs

import org.mule.weave.lsp.utils.URLUtils
import org.mule.weave.v2.editor.VirtualFile
import org.mule.weave.v2.editor.VirtualFileSystem
import org.mule.weave.v2.parser.ast.variables.NameIdentifier
import org.mule.weave.v2.sdk.NameIdentifierHelper
import org.mule.weave.v2.sdk.WeaveResource
import org.mule.weave.v2.sdk.WeaveResourceResolver

import java.io.File
import java.io.InputStream
import java.net.URI
import java.util
import java.util.logging.Level
import java.util.logging.Logger
import java.util.zip.ZipEntry
import java.util.zip.ZipFile
import scala.collection.JavaConverters.asJavaIteratorConverter
import scala.collection.JavaConverters.enumerationAsScalaIteratorConverter
import scala.io.BufferedSource
import scala.io.Source

/**
  * Represents a file system based on a JAR
  *
  * @param jarFile The Jar that backups this file
  */
class JarVirtualFileSystem(override val artifactId: String, val jarFile: File, jarFileNameIdentifierResolver: JarFileNameIdentifierResolver) extends ReadOnlyVirtualFileSystem with ArtifactVirtualFileSystem with AutoCloseable {

  private val logger: Logger = Logger.getLogger(getClass.getName)
  lazy val zipFile = new ZipFile(jarFile)

  override def file(path: String): VirtualFile = {
    logger.log(Level.INFO, s"Before creating uri, entry ${path} in ${jarFile.getAbsolutePath}")
    URLUtils.toURI(path) match {
      case Some(uri) if (uri.getScheme == jarFileNameIdentifierResolver.jarScheme && URLUtils.uriPath(jarFile.getCanonicalPath.replace(File.separatorChar, '/')) == jarPath(uri)) => {
        val str = jarEntry(uri)
        logger.log(Level.INFO, s"Jar entry entry ${str} in ${jarFile.getAbsolutePath}")
        resolveLocalPath(str)
      }
      case _ => {
        logger.log(Level.INFO, s"No jar entry entry ${path} in ${jarFile.getAbsolutePath}")
        resolveLocalPath(path)
      }
    }
  }

  private def jarPath(uri: URI) = {
    val parts: Array[String] = jarUriParts(uri)
    parts(0)
  }

  private def jarEntry(uri: URI) = {
    val parts: Array[String] = jarUriParts(uri)
    parts(1)
  }

  private def jarUriParts(uri: URI) = {
    uri.getPath.split("!")
  }

  private def resolveLocalPath(path: String) = {
    logger.log(Level.FINE, s"file $path in ${jarFile.getAbsolutePath}")
    val zipEntryPath = if (path.startsWith("/")) {
      path.substring(1)
    } else {
      path
    }
    val zipEntry: ZipEntry = zipFile.getEntry(zipEntryPath)
    Option(zipEntry) match {
      case Some(entry) => {
        logger.log(Level.INFO, s"zip entry ${entry.getName} in ${jarFile.getAbsolutePath}")
        new JarVirtualFile(zipEntry.getName, entry, zipFile, this, jarFileNameIdentifierResolver)
      }
      case None => null
    }
  }

  override def listFiles(): util.Iterator[VirtualFile] = {
    val entries = zipFile.entries().asScala
    val list: Iterator[VirtualFile] = entries.flatMap((zipEntry) => {
      if (zipEntry.isDirectory) {
        None
      } else {
        logger.log(Level.FINE, s"Building virtual file ${zipEntry.getName} in ${jarFile.getAbsolutePath}")
        Some(new JarVirtualFile(zipEntry.getName, zipEntry, zipFile, this, jarFileNameIdentifierResolver))
      }
    })
    list.asJava
  }

  override def asResourceResolver: WeaveResourceResolver = {
    new JarVirtualFSResourceProvider(this)
  }

  override def close(): Unit = zipFile.close()
}

class JarVirtualFSResourceProvider(vfs: VirtualFileSystem) extends WeaveResourceResolver {

  override def resolve(name: NameIdentifier): Option[WeaveResource] = {
    val path = NameIdentifierHelper.toWeaveFilePath(name, "/")
    resolvePath(path)
  }

  override def resolvePath(path: String): Option[WeaveResource] = {
    val virtualFile: VirtualFile = vfs.file(path)
    Option(virtualFile).map((_) => {
      virtualFile.asResource()
    })
  }

  override def resolveAll(name: NameIdentifier): Seq[WeaveResource] = {
    resolve(name).toSeq
  }

}


class JarVirtualFile(val entryJar: String, var entry: ZipEntry, zipFile: ZipFile, val fs: JarVirtualFileSystem, jarFileNameIdentifierResolver: JarFileNameIdentifierResolver) extends VirtualFile {

  private lazy val content: String = {
    val stream: InputStream = zipFile.getInputStream(entry)
    val source: BufferedSource = Source.fromInputStream(stream, "UTF-8")
    try {
      source.mkString
    } catch {
      case io: Exception => {
        throw new RuntimeException(s"Exception when trying to read: `${entryJar}`. From file: `${fs.jarFile.getAbsolutePath}`.", io)
      }
    } finally {
      stream.close()
    }
  }

  override def read(): String = {
    content
  }

  override def write(content: String): Boolean = {
    false
  }

  override def readOnly(): Boolean = true

  override def url(): String = {
    val canonicalPath = fs.jarFile.getCanonicalPath
    val canonicalPathReplaced = canonicalPath.replace(File.separatorChar, '/')
    val jarUri = fs.jarFile.toURI
    val str = URLUtils.uriPath(canonicalPathReplaced + "!/" + entryJar)
    val uri = new URI(jarFileNameIdentifierResolver.jarScheme, jarUri.getUserInfo, jarUri.getHost, jarUri.getPort, str, jarUri.getQuery, jarUri.getFragment)
    uri.toString
  }

  override def asResource(): WeaveResource = {
    WeaveResource(url(), read())
  }

  override def getNameIdentifier: NameIdentifier = {
    NameIdentifierHelper.fromWeaveFilePath(entryJar, "/")
  }

  override def path(): String = {
    "/" + entryJar
  }
}
