package org.mule.weave.lsp.utils
import net.liftweb.json.DefaultFormats
import net.liftweb.json.parseOpt
import org.eclipse.lsp4j
import org.eclipse.lsp4j.Position
import org.mule.weave.lsp.agent.WeaveAgentService
import org.mule.weave.lsp.extension.client.PreviewResult
import org.mule.weave.lsp.extension.client.WeaveTestItem
import org.mule.weave.lsp.services.ClientLogger
import org.mule.weave.lsp.services.DataWeaveTestService
import org.mule.weave.v2.editor.VirtualFile
import org.mule.weave.v2.editor.VirtualFileSystem
import org.mule.weave.v2.parser.ast.variables.NameIdentifier

import scala.collection.JavaConverters._

import scala.util.Try

case class WeaveSemanticTestIndexer(agentService: WeaveAgentService, clientLogger: ClientLogger) extends WeaveTestIndexer {
  private var compatibleDWTF: Boolean = true

  override def index(virtualFile: VirtualFile): Option[WeaveTestItem] = {
    if (compatibleDWTF) {
      doIndex(virtualFile)
    } else {
      None
    }
  }

  def disableSemanticIndexing(): Unit = {
    compatibleDWTF = false
    clientLogger.logDebug(s"Disabling semantic indexing")
  }

  private def checkSkippedTestResult(root: TestResult): Boolean = {
    root.tests match {
      case Some(Seq()) | None => root.status == "SKIP"
      case Some(children) => children.forall(checkSkippedTestResult)
    }
  }

  private def checkLocationPresent(root: TestResult): Boolean = {
    root.location.isDefined && root.tests.forall(_.forall(checkLocationPresent))
  }

  private def doIndex(virtualFile: VirtualFile): Option[WeaveTestItem] = {
    val previewResult: PreviewResult = agentService.run(
      virtualFile.getNameIdentifier,
      virtualFile.read(),
      virtualFile.url(),
      None,
      Map("skipAll" -> true.toString),
      Some("application/json"),
      Some(DataWeaveTestService.SINGLE_TEST_FILE_DISCOVER_TIME_OUT),
      withTestsClasspath = true
    )

    if (previewResult.success) {
      val content = previewResult.content
      implicit val formats: DefaultFormats.type = DefaultFormats

      val testResults = parseOpt(content).flatMap(_.extractOpt[TestResult])

      testResults match {
        case Some(value) if !checkSkippedTestResult(value) || !checkLocationPresent(value) =>
          disableSemanticIndexing()
          None
        case Some(value) => value.toTestItem(virtualFile.fs(), virtualFile.url())
        case None => None
      }
    } else {
      clientLogger.logDebug(s"Semantic indexing of ${virtualFile.url()} failed:\n${previewResult.errorMessage}")
      None
    }
  }
}


case class TestResult(name: String, time: Number, status: String, tests: Option[Seq[TestResult]], errorMessage: Option[String], location: Option[TestLocation]) {
  def toTestItem(vfs: VirtualFileSystem, uri: String): Option[WeaveTestItem] = {
    val children = tests.getOrElse(Seq()).flatMap(_.toTestItem(vfs, uri))

    val maybeRange: Option[lsp4j.Range] = for {
      testLocation <- location
      source <- testLocation.sourceIdentifier
      weaveResource <- vfs.asResourceResolver.resolve(NameIdentifier(source))
      if URLUtils.toURI(weaveResource.url()) == URLUtils.toURI(uri)
      range <- testLocation.toRange
    } yield {
      range
    }

    val f = WeaveTestItem(label = name, uri = uri, range = maybeRange.orNull, children = children.asJava)
    Some(f)
  }
}

case class TestLocation(start: Option[TestPosition], end: Option[TestPosition], locationString: String, text: Option[String], sourceIdentifier: Option[String]) {
  def toRange: Option[lsp4j.Range] = {
    for {
      start <- start
      end <- end
      startPos <- start.toPosition
      endPos <- end.toPosition
    } yield {
      new lsp4j.Range(startPos, endPos)
    }
  }
}

case class TestPosition(index: Option[String], line: Option[String], column: Option[String]) {
  def toPosition: Option[Position] = {
    for {
      lineS <- line
      line <- Try(lineS.toInt).toOption
      columnS <- column
      column <- Try(columnS.toInt).toOption
    } yield {
      new Position(line, column)
    }
  }
}
