package org.mule.weave.lsp.commands

import org.eclipse.lsp4j.ExecuteCommandParams
import org.eclipse.lsp4j.MessageParams
import org.eclipse.lsp4j.MessageType
import org.mule.weave.lsp.agent.WeaveAgentService
import org.mule.weave.lsp.extension.client.OpenTextDocumentParams
import org.mule.weave.lsp.extension.client.PreviewResult
import org.mule.weave.lsp.extension.client.WeaveInputBoxParams
import org.mule.weave.lsp.extension.client.WeaveLanguageClient
import org.mule.weave.lsp.project.Project
import org.mule.weave.lsp.services.AgentBasedDataFormatDescriptorProviderService
import org.mule.weave.lsp.services.DataWeaveToolingService
import org.mule.weave.lsp.services.WeaveScenarioManagerService
import org.mule.weave.lsp.utils.URLUtils.toLSPUrl
import org.mule.weave.v2.completion.DataFormatDescriptor
import org.mule.weave.v2.editor.MappingInput
import org.mule.weave.v2.parser.ast.variables.NameIdentifier
import org.mule.weave.v2.scaffolding.ScaffoldingConfiguration
import org.mule.weave.v2.scaffolding.ScaffoldingService
import org.mule.weave.v2.ts.WeaveType

import java.io.File

class CreateInputSampleCommand(
                                weaveLanguageClient: WeaveLanguageClient,
                                toolingService: DataWeaveToolingService,
                                agent: WeaveAgentService,
                                dataFormatsService: AgentBasedDataFormatDescriptorProviderService,
                                scenariosManager: WeaveScenarioManagerService,
                                project: Project
                              ) extends WeaveCommand {

  override def commandId(): String = Commands.DW_CREATE_INPUT_SAMPLE

  override def execute(params: ExecuteCommandParams): AnyRef = {
    val nameIdentifier: String = Commands.argAsString(params.getArguments, 0)
    val nameOfScenario: String = Commands.argAsString(params.getArguments, 1)
    val uri: Option[String] = Commands.optionalArgAsString(params.getArguments, 2)
    val inputName: Option[String] = Commands.optionalArgAsString(params.getArguments, 3)
    val mayBeInputFileName: Option[String] = Commands.optionalArgAsString(params.getArguments, 4)

    if (mayBeInputFileName.isEmpty) {
      val sampleName = weaveLanguageClient.weaveInputBox(WeaveInputBoxParams(title = "Specify The Sample Name", value = "payload.json", prompt = "Name of sample i.e payload.json or vars.foo.csv")).get()
      if (!sampleName.cancelled) {
        val inputFileName: String = sampleName.value
        createInput(nameIdentifier, nameOfScenario, inputFileName, uri, inputName)
      }
    } else {
      createInput(nameIdentifier, nameOfScenario, mayBeInputFileName.get, uri, inputName)
    }
    null
  }

  private def createInput(nameIdentifier: String, nameOfScenario: String, inputFileName: String, uri: Option[String], inputName: Option[String]): Unit = {
    val content: Option[String] =
      if (inputName.isEmpty || uri.isEmpty) {
        None
      } else {
        val maybeInputDirective: Option[MappingInput] = toolingService.inputOf(uri.get, inputName.get)
        val inputType: Option[WeaveType] = maybeInputDirective.flatMap(_.wtype)
        val inputMimeType: Option[String] = maybeInputDirective.flatMap(_.dataFormat.map(_.value))
        if (inputType.isDefined && inputMimeType.isDefined) {
          val dataFormatValue = inputMimeType.get
          val scaffoldingService: ScaffoldingService = new ScaffoldingService()
          val scaffoldingConfiguration: ScaffoldingConfiguration = ScaffoldingConfiguration(project.settings.scaffoldingElementNumber.intValue())

          val maybeFormatDescriptor: Option[DataFormatDescriptor] =
            dataFormatsService
              .dataFormats()
              .find((d) => d.mimeType.equals(dataFormatValue) || d.id.exists(_.equals(dataFormatValue)))

          val writerOptions = maybeFormatDescriptor match {
            case Some(formatDescriptor) => {
              maybeInputDirective.map((id) => {
                id.properties
                  .filter((p) => formatDescriptor.writerProperties.exists(_.name == p.name))
                  .map((p) => {
                    val value = p.value
                    val descriptorProp = formatDescriptor.writerProperties.find(_.name == p.name).get
                    descriptorProp.wtype match {
                      case "String" => (p.name, value)
                      case "Boolean" => (p.name, value.toBoolean)
                      case "Number" => (p.name, value.toInt)
                      case _ => (p.name, value)
                    }
                  }).toMap
              }).getOrElse(Map())
            }
            case None => Map[String, Any]()
          }

          val scaffolding: String = scaffoldingService.scaffold(inputType.get, dataFormatValue, writerOptions, scaffoldingConfiguration)
          val result: PreviewResult = agent.run(NameIdentifier.ANONYMOUS_NAME, scaffolding, "fake://scaffold.dwl")
          if (result.success) {
            Some(result.content)
          } else {
            None
          }
        } else {
          None
        }
      }

    val maybeFile: Option[File] = scenariosManager.createScenario(NameIdentifier(nameIdentifier), nameOfScenario, inputFileName, content)
    if (maybeFile.isDefined) {
      weaveLanguageClient.openTextDocument(OpenTextDocumentParams(toLSPUrl(maybeFile.get)))
    } else {
      weaveLanguageClient.showMessage(new MessageParams(MessageType.Error, s"Unable to create sample data file ${inputFileName}."))
    }
  }

  override def name(): String = "Create Sample Data."

  override def description(params: ExecuteCommandParams): String = s"Creating Sampling Data for: ${Commands.argAsString(params.getArguments, 0)}."
}
