package org.mule.weave.extension.api.llm.tools;

import java.util.ArrayList;
import java.util.List;

/**
 * Builder for {@link LLMToolResponse}.
 * <p>
 * This class provides a fluent API to construct {@link LLMToolResponse} instances.
 * It allows for incrementally adding messages and setting the success status.
 * By default, a new builder instance will have a success status of {@code true}
 * and an empty list of messages.
 * </p>
 *
 * @see LLMToolResponse
 */
public class LLMToolResponseBuilder {
    private boolean success = true;
    private final List<String> messages = new ArrayList<>();

    /**
     * Adds a single message to the list of messages for the {@link LLMToolResponse}.
     *
     * @param message The message string to add.
     * @return This {@link LLMToolResponseBuilder} instance for fluent chaining.
     */
    public LLMToolResponseBuilder withMessage(String message) {
        this.messages.add(message);
        return this;
    }

    /**
     * Sets the success status for the {@link LLMToolResponse}.
     *
     * @param success {@code true} if the operation was successful, {@code false} otherwise.
     * @return This {@link LLMToolResponseBuilder} instance for fluent chaining.
     */
    public LLMToolResponseBuilder withSuccess(boolean success) {
        this.success = success;
        return this;
    }

    /**
     * Adds a list of messages to the existing messages for the {@link LLMToolResponse}.
     *
     * @param messages A list of message strings to add.
     * @return This {@link LLMToolResponseBuilder} instance for fluent chaining.
     */
    public LLMToolResponseBuilder withMessages(List<String> messages) {
        this.messages.addAll(messages);
        return this;
    }

    /**
     * Builds and returns the {@link LLMToolResponse} instance with the accumulated
     * messages and the set success status.
     *
     * @return A new {@link LLMToolResponse} instance.
     */
    public LLMToolResponse build() {
        return new LLMToolResponse(messages, success);
    }
}