package org.mule.weave.extension.api;

import org.eclipse.lsp4j.Location;
import org.eclipse.lsp4j.Range;
import org.eclipse.lsp4j.services.LanguageServer;

public interface WeaveLanguageServerBuilder {

    /**
     * ID to identify the created builder. Used for logging reasons.
     *
     * @param id The ID to for this LS
     * @return this builder.
     */
    WeaveLanguageServerBuilder withId(String id);

    /**
     * If the created LS should contribute with code lenses.
     *
     * @param codeLenses whether the created LS should contribute with code lenses.
     * @return this builder.
     */
    WeaveLanguageServerBuilder supportsCodeLenses(Boolean codeLenses);

    /**
     * If the created LS should contribute with code actions.
     *
     * @param codeActions whether the created LS should contribute with code actions.
     * @return this builder.
     */
    WeaveLanguageServerBuilder supportsCodeActions(Boolean codeActions);

    /**
     * The file scheme that this language server will handle
     *
     * @param scheme The file schema
     * @return this builder.
     */
    WeaveLanguageServerBuilder fileScheme(String scheme);

    /**
     * The locator for this language server to handle the file scheme.
     * Needed if a file scheme is defined.
     *
     * @param textDocumentLocator The text document locator.
     * @return this builder.
     */
    WeaveLanguageServerBuilder textDocumentTranslator(TextDocumentTranslator textDocumentLocator);

    /**
     * Builds the LS
     *
     * @return a new Language Server
     */
    LanguageServer build();

    /**
     * The Language Server would handle DataWeave files based on {@link WeaveLanguageServerBuilder#fileScheme(String)}
     * so a {@link TextDocumentTranslator} is responsible for translating the embedded {@link java.net.URI} format to
     * a {@link Location} pointing to the file system textDocument file.
     */
    interface TextDocumentTranslator {

        /**
         * For embedded URI (as {@link String}) it does
         * the translation to the textDocument that holds this embedded uri.
         *
         * @param embeddedUri an embedded URI.
         * @return the URL of the textDocument.
         */
        String translate(String embeddedUri);

        /**
         * For embedded URI (as {@link String}) and {@link Range} located related to the content of the script, it does
         * the translation to the textDocument and the correct range coordinates.
         *
         * @param embeddedUri an embedded URI.
         * @param range a range relative to the embedded URI.
         * @return the container textDocument location.
         */
        Location translate(String embeddedUri, Range range);

    }

}
