package org.mule.weave.extension.api.services.sample;

import org.mule.weave.extension.api.services.WeaveService;

import java.io.File;
import java.util.List;
import java.util.Objects;
import java.util.Optional;

public interface WeaveSampleDataResourcesService extends WeaveService {

    /**
     * The folder were sample data should be stored.
     *
     * @param uri for the unique identifier of the element to get its sample data folder.
     * @return {@link File} the folder were the sample data would be created. The folder is not created in the
     * file system it is up to consumer to create it.
     */
    File getSampleDataFolder(String uri);

    /**
     * The file where the input sample data should be stored.
     * In order to check if input sample data exists use the {@link File#exists()} API.
     *
     * @param uri           for the unique identifier of the element to get its sample data folder.
     * @param scenarioName  the name of the scenario where the input is defined.
     * @param inputFileName the file name (with its extension, .json, .xml).
     * @return {@link File} the file were the input sample data should be stored. The file is not created in the
     * file system it is up to consumer to create it.
     */
    File getInputSampleData(String uri, String scenarioName, String inputFileName);

    /**
     * If there is a scenario defined for the URI it lists the {@link SampleInput} for every input sample defined or an empty list.
     * @param uri           for the unique identifier of the element to get its sample data folder.
     * @param scenarioName  the name of the scenario where the inputs are defined.
     * @return {@link List} of {@link SampleInput}s for every input sample defined, or empty list. If there is no
     *         scenario defined for the given name an empty Optional.
     */
    Optional<List<SampleInput>> getInputSamples(String uri, String scenarioName);

    /**
     * For the given URI returns the list of names for each scenario defined.
     * @param uri for the unique identifier of the element to get its sample data folder..
     * @return {@link List} of scenario names.
     */
    List<String> listScenarios(String uri);

    /**
     * DTO that represents an input sample data with a name and its uri.
     */
    class SampleInput {
        private final String uri;
        private final String name;

        public SampleInput(String uri, String name) {
            Objects.requireNonNull(uri, "'uri' cannot be null");
            Objects.requireNonNull(name,  "'name' cannot be null");
            this.uri = uri;
            this.name = name;
        }

        public String getUri() {
            return uri;
        }

        public String getName() {
            return name;
        }

        @Override
        public boolean equals(Object o) {
            if (this == o) return true;
            if (!(o instanceof SampleInput)) return false;
            SampleInput that = (SampleInput) o;
            return Objects.equals(uri, that.uri) && Objects.equals(name, that.name);
        }

        @Override
        public int hashCode() {
            return Objects.hash(uri, name);
        }
    }
}
