package org.mule.weave.extension.api.component.structure;

import java.io.File;
import java.util.ArrayList;
import java.util.List;

public class WeaveProjectStructureBuilder {
    private final List<ModuleStructureBuilder> modules = new ArrayList<>();

    public ModuleStructureBuilder addModule() {
        ModuleStructureBuilder moduleStructureBuilder = new ModuleStructureBuilder();
        this.modules.add(moduleStructureBuilder);
        return moduleStructureBuilder;
    }

    public WeaveProjectStructure build() {
        return new WeaveProjectStructure(modules.stream().map(ModuleStructureBuilder::build).toArray(WeaveModuleStructure[]::new));
    }

    public static class RootStructureBuilder {

        private WeaveRootKind kind = WeaveRootKind.MAIN;
        private final List<File> sources = new ArrayList<>();
        private final List<File> resources = new ArrayList<>();
        private File defaultSourceFolder;


        public RootStructureBuilder withKind(WeaveRootKind kind) {
            this.kind = kind;
            return this;
        }

        public RootStructureBuilder addSource(File source) {
            if (defaultSourceFolder == null) {
                this.defaultSourceFolder = source;
            }
            this.sources.add(source);
            return this;
        }

        public RootStructureBuilder addResource(File source) {
            if (defaultSourceFolder == null) {
                this.defaultSourceFolder = source;
            }
            this.resources.add(source);
            return this;
        }

        public RootStructureBuilder withDefaultSourceFolder(File source) {
            this.defaultSourceFolder = source;
            return this;
        }


        public WeaveRootStructure build() {
            return new WeaveRootStructure(kind, sources.toArray(File[]::new), resources.toArray(File[]::new), defaultSourceFolder);
        }

        public void addSources(List<File> srcFolders) {
            for (File srcFolder : srcFolders) {
                addSource(srcFolder);
            }
        }

        public void addResources(List<File> resourceFolders) {
            for (File resourceFolder : resourceFolders) {
                addResource(resourceFolder);
            }
        }
    }

    public static class ModuleStructureBuilder {

        private String name;
        private final List<RootStructureBuilder> roots = new ArrayList<>();
        private final List<WeaveTargetFolderBuilder> targets = new ArrayList<>();
        private final List<WeaveDescriptorFile> descriptors = new ArrayList<>();

        public ModuleStructureBuilder withName(String name) {
            this.name = name;
            return this;
        }

        public ModuleStructureBuilder addDescriptor(String kind, File descriptor) {
            this.descriptors.add(new WeaveDescriptorFile(kind, descriptor));
            return this;
        }

        public RootStructureBuilder addRoot() {
            RootStructureBuilder e = new RootStructureBuilder();
            this.roots.add(e);
            return e;
        }

        public WeaveTargetFolderBuilder addTarget() {
            WeaveTargetFolderBuilder e = new WeaveTargetFolderBuilder();
            this.targets.add(e);
            return e;
        }

        public WeaveModuleStructure build() {
            if (name == null) {
                throw new RuntimeException("Missing name field");
            }

            return new WeaveModuleStructure(name,
                    roots.stream().map(RootStructureBuilder::build).toArray(WeaveRootStructure[]::new),
                    targets.stream().map(WeaveTargetFolderBuilder::build).toArray(WeaveTargetFolder[]::new),
                    descriptors.toArray(WeaveDescriptorFile[]::new));
        }

    }

    public static class WeaveTargetFolderBuilder {
        private WeaveTargetKind kind = WeaveTargetKind.RESOURCES;
        private final List<File> files = new ArrayList<>();

        public WeaveTargetFolderBuilder withKind(WeaveTargetKind kind) {
            this.kind = kind;
            return this;
        }

        public WeaveTargetFolderBuilder addTargetDir(File dir) {
            this.files.add(dir);
            return this;
        }

        public WeaveTargetFolder build() {
            return new WeaveTargetFolder(kind, files.toArray(File[]::new));
        }
    }
}
