package org.mule.weave.lsp

import org.eclipse.lsp4j.CodeLensOptions
import org.eclipse.lsp4j.CompletionOptions
import org.eclipse.lsp4j.ExecuteCommandOptions
import org.eclipse.lsp4j.InitializeParams
import org.eclipse.lsp4j.InitializeResult
import org.eclipse.lsp4j.InitializedParams
import org.eclipse.lsp4j.ServerCapabilities
import org.eclipse.lsp4j.ServerInfo
import org.eclipse.lsp4j.SignatureHelpOptions
import org.eclipse.lsp4j.TextDocumentSyncKind
import org.eclipse.lsp4j.jsonrpc.services.JsonDelegate
import org.eclipse.lsp4j.services.LanguageClient
import org.eclipse.lsp4j.services.LanguageServer
import org.eclipse.lsp4j.services.WorkspaceService
import org.mule.dx.platform.api.protocol.ClientAware
import org.mule.weave.lsp.actions.CodeActionsManagerService
import org.mule.weave.lsp.commands.CommandManagerService
import org.mule.weave.lsp.extension.client.LanguageClientDelegate
import org.mule.weave.lsp.extension.protocol.DataWeaveProtocolService
import org.mule.weave.lsp.extension.services.WeaveTextDocumentService
import org.mule.weave.lsp.indexer.LSPWeaveIndexService
import org.mule.weave.lsp.jobs.JobManagerService
import org.mule.weave.lsp.jobs.Status
import org.mule.weave.lsp.project.DefaultProjectMetadata
import org.mule.weave.lsp.project.ProjectKind
import org.mule.weave.lsp.project.ProjectKindContext
import org.mule.weave.lsp.project.ProjectKindFactory
import org.mule.weave.lsp.services.DiagnosticsPublisher
import org.mule.weave.lsp.services.DiagnosticsPublisherService
import org.mule.weave.lsp.services._
import org.mule.weave.lsp.services.delegate.TextDocumentServiceDelegate
import org.mule.weave.lsp.services.delegate.WorkspaceServiceDelegate
import org.mule.weave.lsp.vfs.ProjectFileSystemService
import org.mule.weave.v2.editor.VirtualFileSystem

import java.util
import java.util.concurrent.CompletableFuture
import java.util.concurrent.ExecutorService
import java.util.logging.Level
import java.util.logging.Logger
import scala.collection.JavaConverters.seqAsJavaListConverter

class WeaveLanguageServer(projectKindFactory: ProjectKindFactory, configuration: LanguageServerConfiguration, service: DataWeaveProtocolService) extends LanguageServer with ClientAware[LanguageClient] {

  private val logger: Logger = Logger.getLogger(getClass.getName)
  private val workspaceService: WorkspaceServiceDelegate = new WorkspaceServiceDelegate()
  private val textDocumentService: TextDocumentServiceDelegate = new TextDocumentServiceDelegate()

  private val languageClient: LanguageClientDelegate = new LanguageClientDelegate()

  private var projectMetadata: DefaultProjectMetadata = _
  private var clientLogger: ClientLogger = _

  protected var projectKind: ProjectKind = _

  def project(): DefaultProjectMetadata = projectMetadata

  override def initialize(params: InitializeParams): CompletableFuture[InitializeResult] = {
    logger.log(Level.INFO, s"Initialize (connected?: ${languageClient.delegate != null})")
    CompletableFuture.supplyAsync(() => {
      logger.log(Level.INFO, s"Initializing... (connected?: ${languageClient.delegate != null})")
      Runtime.getRuntime.addShutdownHook(new Thread(() => this.shutdown().get()))
      val clientLoggerFactory = new ClientLoggerFactory(languageClient)
      clientLogger = clientLoggerFactory.createLogger(classOf[WeaveLanguageServer], configuration.lspID)
      projectMetadata = DefaultProjectMetadata.create(params)
      // Create the project
      projectKind = projectKindFactory.createKind(
        ProjectKindContext(
          projectMetadata,
          clientLoggerFactory,
          service.getClient(),
          languageClient
        )
      )

      //TODO: W-17148935 bug in DX to delegate mediator's call to wrapped languageClient
      if (configuration.fileScheme.isEmpty) {
        projectKind.toolingService(classOf[ConfigurationService]).setLanguageClient(languageClient)
        projectKind.toolingService(classOf[UIService]).setLanguageClient(languageClient)
      }

      val publisherService = projectKind.toolingService(classOf[DiagnosticsPublisherService])
      val workspaceEditsService = projectKind.toolingService(classOf[WorkspaceEditService])
      if (configuration.fileScheme.isDefined) {
        publisherService.addPublisher(new DiagnosticsPublisher(languageClient, configuration.fileScheme))
        workspaceEditsService.addApplier(new WorkspaceEditApplier(languageClient, configuration.fileScheme))
      } else {
        publisherService.setDefaultPublisher(new DiagnosticsPublisher(languageClient, Option.empty))
        workspaceEditsService.setDefaultApplier(new WorkspaceEditApplier(languageClient, Option.empty))
      }

      //Inject the project event bus
      projectMetadata.setEventBus(projectKind.eventBus())

      val documentServiceImpl =
        new DataWeaveDocumentService(
          projectKind.toolingService(classOf[DataWeaveToolingService]),
          executor(),
          projectKind.toolingService(classOf[ProjectFileSystemService]),
          projectKind.toolingService(classOf[WeaveScenarioManagerService]),
          projectKind.vfs(),
          projectKind.toolingService(classOf[CodeActionsManagerService]),
          configuration.codeLens
        )


      textDocumentService.delegate = documentServiceImpl

      val workspaceServiceImpl =
        new DataWeaveWorkspaceService(
          projectMetadata,
          projectKind.vfs(),
          clientLoggerFactory,
          service.getClient(),
          projectKind.toolingService(classOf[JobManagerService]),
          projectKind.toolingService(classOf[LSPWeaveIndexService]),
          projectKind.toolingService(classOf[CommandManagerService])
        )

      workspaceService.delegate = workspaceServiceImpl

      documentServiceImpl.initialize(projectKind = projectKind, eventBus = projectKind.eventBus())
      workspaceServiceImpl.initialize(projectKind = projectKind, eventBus = projectKind.eventBus())
      projectKind.initialize()

      val commandManagerService = projectKind.toolingService(classOf[CommandManagerService])
      val capabilities = new ServerCapabilities
      capabilities.setTextDocumentSync(TextDocumentSyncKind.Full)
      capabilities.setCompletionProvider(new CompletionOptions(true, java.util.Arrays.asList(".")))
      capabilities.setHoverProvider(true)
      capabilities.setDocumentSymbolProvider(true)
      capabilities.setDefinitionProvider(true)
      capabilities.setDocumentFormattingProvider(true)
      capabilities.setFoldingRangeProvider(true)
      capabilities.setCodeActionProvider(configuration.codeActions)
      capabilities.setWorkspaceSymbolProvider(true)
      capabilities.setCodeLensProvider(new CodeLensOptions(configuration.codeLens))
      capabilities.setRenameProvider(true)
      capabilities.setDocumentRangeFormattingProvider(true)
      capabilities.setReferencesProvider(true)
      capabilities.setSignatureHelpProvider(new SignatureHelpOptions(util.Arrays.asList("("), util.Arrays.asList(",")))
      if (configuration.commands) {
        capabilities.setExecuteCommandProvider(new ExecuteCommandOptions(commandManagerService.commandIds().asJava))
      }
      val serverInfo = new ServerInfo("DataWeave", projectKind.getWeaveVersion())
      clientLogger.logInfo(s"Initialize: Finished. Server: $serverInfo, Capabilities: $capabilities")
      new InitializeResult(capabilities, serverInfo)
    })
  }

  override def initialized(params: InitializedParams): Unit = {
    //Start the project
    projectKind.toolingService(classOf[JobManagerService])
      .schedule((_: Status) => {
        try {
          clientLogger.logInfo(s"Project Kind: ${projectKind.name()} Starting.")
          projectKind.initialized()
          //Mark project as initialized
          projectMetadata.markStarted()
          clientLogger.logInfo("ProjectStartedEvent fired")
        } catch {
          case e: Exception =>
            clientLogger.logInfo(s"Unable to Start project.", e)
        }
      }, s"Starting DataWeave Project", s"Starting Project of Kind: ${projectKind.name()}.")
  }

  override def shutdown(): CompletableFuture[AnyRef] = {
    CompletableFuture.supplyAsync(() => {
      clientLogger.logInfo("Stopping the services")
      projectKind.shutdown()
      null
    })
  }

  override def exit(): Unit = {
    System.exit(0)
  }

  @JsonDelegate
  override def getTextDocumentService: WeaveTextDocumentService = {
    logger.log(Level.INFO, "getTextDocumentService")
    textDocumentService
  }

  override def getWorkspaceService: WorkspaceService = {
    logger.log(Level.INFO, "getWorkspaceService")
    workspaceService
  }

  override def connect(client: LanguageClient): Unit = {
    logger.log(Level.INFO, "connect")
    languageClient.delegate = client
  }

  /**
   * Returns the File System for this language server
   *
   * @return Returns the VFS
   */
  def fs(): VirtualFileSystem = {
    projectKind.vfs()
  }

  def executor(): ExecutorService = {
    IDEExecutors.defaultExecutor()
  }
}

case class LanguageServerConfiguration(
                                        lspID: String = "DATA_WEAVE_LSP",
                                        commands: Boolean = true,
                                        codeActions: Boolean = true,
                                        codeLens: Boolean = true,
                                        fileScheme: Option[String] = None
                                      ) {}

object LanguageServerConfiguration {
  def create(): LanguageServerConfiguration = LanguageServerConfiguration()
}

