package org.mule.weave.lsp.commands

import org.mule.weave.extension.api.extension.command.WeaveCommand
import org.mule.weave.extension.api.project.ProjectMetadata
import org.mule.weave.lsp.agent.WeaveAgentService
import org.mule.weave.lsp.jobs.JobManagerService
import org.mule.weave.lsp.project.DefaultProjectMetadata
import org.mule.weave.lsp.project.ProjectKind
import org.mule.weave.lsp.services.AgentBasedDataFormatDescriptorProviderService
import org.mule.weave.lsp.services.ClientLoggerFactory
import org.mule.weave.lsp.services.DataWeaveTestService
import org.mule.weave.lsp.services.DataWeaveToolingService
import org.mule.weave.lsp.services.PreviewService
import org.mule.weave.lsp.services.ToolingService
import org.mule.weave.lsp.services.WeaveScenarioManagerService
import org.mule.weave.lsp.utils.InternalEventBus
import org.mule.weave.lsp.vfs.ProjectFileSystemService
import org.mule.weave.v2.editor.VirtualFileSystem

class CommandManagerService(virtualFileSystem: VirtualFileSystem,
                            projectVirtualFileSystem: ProjectFileSystemService,
                            loggerFactory: ClientLoggerFactory,
                            project: ProjectMetadata,
                            jobManagerService: JobManagerService,
                            weaveToolingService: DataWeaveToolingService,
                            weaveAgentService: WeaveAgentService,
                            scenariosManager: WeaveScenarioManagerService,
                            previewService: PreviewService,
                            dataWeaveTestService: DataWeaveTestService,
                            dataFormatsService: AgentBasedDataFormatDescriptorProviderService
                           ) extends ToolingService {


  override def initialize(projectKind: ProjectKind, eventBus: InternalEventBus): Unit = {
    initProjectCommands(projectKind)
  }

  private def initProjectCommands(projectKind: ProjectKind): Unit = {
    val commands = Seq(
      new RunBatTestCommand(loggerFactory, projectKind, weaveToolingService),
      new RunBatFolderTestCommand(loggerFactory, projectKind),
      new InstallBatCommand(loggerFactory, projectKind),
      new CreateProjectCommand(project, projectKind),
      new CreateTestCommand(projectKind),
      new CreateMappingFileCommand(projectKind),
      new CreateModuleFileCommand(projectKind),
      new CreateIntegrationMappingFileCommand(projectKind, weaveToolingService, virtualFileSystem),
      new RunWeaveCommand(virtualFileSystem, projectVirtualFileSystem, project.asInstanceOf[DefaultProjectMetadata], projectKind, loggerFactory, jobManagerService, dataWeaveTestService),
      new ReloadDependenciesCommand(projectKind),
      new CreateUnitTest(weaveToolingService, projectKind, project),
      new CreateScenarioCommand(scenariosManager, projectKind),
      new RunPreviewCommand(previewService, virtualFileSystem, projectKind),
      new LaunchWeaveCommand(projectKind),
      new LaunchWeaveTestCommand(virtualFileSystem, project, projectKind),
      new QuickFixCommand(weaveToolingService, projectKind),
      new InsertDocumentationCommand(weaveToolingService, projectKind),
      new InsertWeaveTypeCommand(weaveToolingService, project, projectKind),
      new ExtractVariableCommand(weaveToolingService, projectKind),
      new ExtractConstantCommand(weaveToolingService, projectKind),
      new ExtractFunctionCommand(weaveToolingService, projectKind),
      new CreateInputSampleCommand(weaveToolingService, weaveAgentService, dataFormatsService, scenariosManager, project, projectKind),
      new CreateMappingTestCommand(scenariosManager, virtualFileSystem, previewService, projectKind),
    ) ++ projectKind.customCommands()
    projectDependentCommands = Some(commands)
  }

  var projectDependentCommands: Option[Seq[WeaveCommand]] = None

  lazy val projectIndependentCommands: Seq[WeaveCommand] = Seq(
    new DeleteScenarioCommand(scenariosManager),
    new SetActiveScenarioCommand(scenariosManager),
    new DeleteInputSampleCommand(scenariosManager),
    new DeleteOutputSampleCommand(scenariosManager),
    new EnablePreviewModeCommand(previewService, virtualFileSystem),
    new RestartAgentCommand(weaveAgentService)
  )

  def commands: Seq[WeaveCommand] = {
    projectIndependentCommands ++ projectDependentCommands.getOrElse(Seq())
  }

  def commandBy(commandId: String): Option[WeaveCommand] = {
    commands.find(_.commandId() == commandId)
  }

  def commandIds(): Seq[String] = {
    commands.map(_.commandId())
  }

}
