package org.mule.weave.lsp.commands

import org.eclipse.lsp4j
import org.eclipse.lsp4j.ApplyWorkspaceEditParams
import org.eclipse.lsp4j.Command
import org.eclipse.lsp4j.ExecuteCommandParams
import org.eclipse.lsp4j.Position
import org.eclipse.lsp4j.TextEdit
import org.eclipse.lsp4j.WorkspaceEdit
import org.mule.weave.extension.api.extension.command.WeaveCommand
import org.mule.weave.extension.api.project.ProjectMetadata
import org.mule.weave.lsp.project.ProjectKind
import org.mule.weave.lsp.services.DataWeaveToolingService
import org.mule.weave.lsp.services.WorkspaceEditService
import org.mule.weave.v2.editor.WeaveDocumentToolingService
import org.mule.weave.v2.parser.ast.AstNode
import org.mule.weave.v2.parser.ast.functions.FunctionNode
import org.mule.weave.v2.parser.ast.header.directives.DirectiveNode
import org.mule.weave.v2.parser.ast.header.directives.FunctionDirectiveNode
import org.mule.weave.v2.parser.ast.header.directives.InputDirective
import org.mule.weave.v2.parser.ast.header.directives.VarDirective
import org.mule.weave.v2.parser.ast.variables.NameIdentifier
import org.mule.weave.v2.parser.location.WeaveLocation
import org.mule.weave.v2.ts.WeaveType
import org.mule.weave.v2.utils.WeaveTypeEmitterConfig

import java.util

class InsertWeaveTypeCommand(validationService: DataWeaveToolingService, project: ProjectMetadata, projectKind: ProjectKind) extends WeaveCommand {
  override def commandId(): String = Commands.DW_INSERT_WEAVE_TYPE

  override def execute(params: ExecuteCommandParams): AnyRef = {
    val useLiterals = project.settings.useLiteralOnInsertType.booleanValue()
    val args: util.List[AnyRef] = params.getArguments

    val uri: String = Commands.argAsString(args, 0)
    val line: Int = Commands.argAsInt(args, 1)
    val column: Int = Commands.argAsInt(args, 2)

    val documentToolingService: WeaveDocumentToolingService = validationService.openDocument(uri)
    val maybeDirective: Option[AstNode] = documentToolingService.nodeAt(line - 1, column - 1, Some(classOf[DirectiveNode])).map(_.astNode)
    val workspaceEditService = projectKind.toolingService(classOf[WorkspaceEditService])

    maybeDirective match {
      case Some(functionDirectiveNode: FunctionDirectiveNode) => {
        val literal = functionDirectiveNode.literal
        literal match {
          case fv: FunctionNode if (fv.returnType.isEmpty) => {
            val location: WeaveLocation = fv.body.location()
            val weaveType: WeaveType = documentToolingService.typeOf(location.startPosition.index, location.endPosition.index)
            if (weaveType != null) {
              val applyWorkspaceEditParams = new ApplyWorkspaceEditParams()
              val localChanges = new util.HashMap[String, util.List[TextEdit]]()
              val parametersLocation = fv.params.location()
              val position = new Position(parametersLocation.endPosition.line - 1, parametersLocation.endPosition.column - 1)
              val range = new lsp4j.Range(position, position)
              val config = WeaveTypeEmitterConfig(prettyPrint = false, skipMetadataConstraints = true, nameOnly = true, useLiteralType = useLiterals, simplifyTypes = true)
              localChanges.put(uri, util.Arrays.asList(new TextEdit(range, ": " + weaveType.toString(config))))
              applyWorkspaceEditParams.setEdit(new WorkspaceEdit(localChanges))
              workspaceEditService.applyEdit(applyWorkspaceEditParams)
            }
          }
          case _ =>
        }
      }
      case Some(varDirective: VarDirective) if (varDirective.wtype.isEmpty) => {
        val location: WeaveLocation = varDirective.value.location()
        val weaveType: WeaveType = documentToolingService.typeOf(location.startPosition.index, location.endPosition.index)
        if (weaveType != null) {
          val applyWorkspaceEditParams = new ApplyWorkspaceEditParams()
          val localChanges = new util.HashMap[String, util.List[TextEdit]]()
          val parametersLocation = varDirective.variable.location()
          val position = new Position(parametersLocation.endPosition.line - 1, parametersLocation.endPosition.column - 1)
          val range = new lsp4j.Range(position, position)

          val config = WeaveTypeEmitterConfig(prettyPrint = false, nameOnly = true, useLiteralType = useLiterals, simplifyTypes = true)
          localChanges.put(uri, util.Arrays.asList(new TextEdit(range, ": " + weaveType.toString(config))))
          applyWorkspaceEditParams.setEdit(new WorkspaceEdit(localChanges))
          workspaceEditService.applyEdit(applyWorkspaceEditParams)
        }
      }
      case Some(inputDirective: InputDirective) if (inputDirective.wtype.isEmpty) => {
        val location: WeaveLocation = inputDirective.variable.location()
        val weaveType: WeaveType = documentToolingService.typeOf(location.startPosition.index, location.endPosition.index)
        if (weaveType != null) {
          val applyWorkspaceEditParams = new ApplyWorkspaceEditParams()
          val localChanges = new util.HashMap[String, util.List[TextEdit]]()
          val parametersLocation = inputDirective.variable.location()
          val position = new Position(parametersLocation.endPosition.line - 1, parametersLocation.endPosition.column - 1)
          val range = new lsp4j.Range(position, position)

          val config = WeaveTypeEmitterConfig(prettyPrint = false, nameOnly = true, useLiteralType = useLiterals, simplifyTypes = true)
          localChanges.put(uri, util.Arrays.asList(new TextEdit(range, ": " + weaveType.toString(config))))
          applyWorkspaceEditParams.setEdit(new WorkspaceEdit(localChanges))
          workspaceEditService.applyEdit(applyWorkspaceEditParams)
        }
      }
      case _ =>
    }
    null
  }

  override def name(): String = "Inserts DataWeave Type"

  override def description(params: ExecuteCommandParams): String = "Inserting DataWeave Type"
}

object InsertWeaveTypeCommand {

  def createCommand(uri: String, nameIdentifier: NameIdentifier): Command = {
    val nodeLocation: WeaveLocation = nameIdentifier.location()
    new Command("Add Type Annotation",
      Commands.DW_INSERT_WEAVE_TYPE,
      util.Arrays.asList(
        uri,
        nodeLocation.startPosition.line: java.lang.Integer,
        nodeLocation.endPosition.column: java.lang.Integer
      )
    )
  }
}




