package org.mule.weave.lsp.extension.protocol

import java.util.concurrent.CompletableFuture

class DataWeaveProtocolClientDelegate extends DataWeaveProtocolClient {
  var delegate: DataWeaveProtocolClient = NoopDataWeaveProtocolClientDelegate

  def setDelegate(delegate: DataWeaveProtocolClient) = {
    this.delegate = delegate;
  }

  /**
   * Opens an input box to ask the user for input.
   *
   * @return the user provided input. The future can be cancelled, meaning
   *         the input box should be dismissed in the editor.
   */
  override def weaveInputBox(params: WeaveInputBoxParams): CompletableFuture[WeaveInputBoxResult] = delegate.weaveInputBox(params)

  /**
   * Opens an menu to ask the user to pick one of the suggested options.
   *
   * @return the user provided pick. The future can be cancelled, meaning
   *         the input box should be dismissed in the editor.
   */
  override def weaveQuickPick(params: WeaveQuickPickParams): CompletableFuture[WeaveQuickPickResult] = delegate.weaveQuickPick(params)

  override def weaveFileBrowser(params: WeaveOpenDialogParams): CompletableFuture[String] = delegate.weaveFileBrowser(params)

  override def setEditorDecorations(params: EditorDecorationsParams): Unit = delegate.setEditorDecorations(params)

  override def clearEditorDecorations(): Unit = delegate.clearEditorDecorations()

  /**
   * Opens a folder in a new window
   *
   * @param params
   */
  override def openWindow(params: OpenWindowsParams): Unit = delegate.openWindow(params)

  /**
   * This notification is sent from the server to the client to execute the specified configuration on client side.
   *
   */
  override def runConfiguration(config: LaunchConfiguration): Unit = delegate.runConfiguration(config)

  /**
   * This notification is sent from the server to the client to open an editor with the specified document uri
   *
   * @param params The document to be opened
   */
  override def openTextDocument(params: OpenTextDocumentParams): Unit = delegate.openTextDocument(params)

  /**
   * This notification is sent from the server to the client to set context variable on visual studio code
   * used for enabling/disabling commands
   * at the editor with the specified document uri
   *
   * @param params The document to be opened
   */
  override def setContext(params: SetContextParams): Unit = delegate.setContext(params)

  /**
   * This notification is sent from the server to the client to show the live data of a script
   *
   * @param result The result of executing a script
   */
  override def showPreviewResult(result: PreviewResult): Unit = delegate.showPreviewResult(result)

  /**
   * This notification is sent from the server to the client to publish all the resolved dependencies of this workspace
   *
   * @param resolvedDependency The list of all the resolved dependencies
   */
  override def publishDependencies(resolvedDependency: DependenciesParams): Unit = delegate.publishDependencies(resolvedDependency)

  /**
   * This notification is sent from the server to the client to publish current transformation scenarios.
   *
   * @param scenariosParam Scenarios Parameter
   */
  override def showScenarios(scenariosParam: ShowScenariosParams): Unit = delegate.showScenarios(scenariosParam)

  /**
   * This notification is sent from the server to the client to inform the user that a background job has started.
   *
   * @param job The job information that has started
   */
  override def notifyJobStarted(job: JobStartedParams): Unit = {
    if (delegate != null) {
      delegate.notifyJobStarted(job)
    }
  }

  /**
   * This notification is sent from the server to the client to inform the user that a background job has finish.
   *
   * @param job The job information that has ended
   */
  override def notifyJobEnded(job: JobEndedParams): Unit = {
    if (delegate != null) {
      delegate.notifyJobEnded(job)
    }
  }

  /**
   * This notification is sent from the server to the client to push all the possible tests to run on the project.
   *
   * @param job The job information that has ended
   */
  override def publishTestItems(job: PublishTestItemsParams): Unit = {
    if (delegate != null) {
      delegate.publishTestItems(job)
    }
  }

  /**
   * This notification is sent from the server to the client to push tests results.
   *
   * @param job The job information that has ended
   */
  override def publishTestResults(testResults: PublishTestResultsParams): Unit = {
    if (delegate != null) {
      delegate.publishTestResults(testResults)
    }
  }
}
