package org.mule.weave.lsp.project

import org.eclipse.lsp4j.ResourceOperation
import org.eclipse.lsp4j.TextDocumentEdit
import org.eclipse.lsp4j.jsonrpc.messages.Either
import org.mule.weave.extension.api.component.builder.WeaveBuildComponent
import org.mule.weave.extension.api.component.dependency.WeaveDependencyComponent
import org.mule.weave.extension.api.component.embedded.WeaveTextDocumentLocator
import org.mule.weave.extension.api.component.metadata.WeaveMetadataProviderComponent
import org.mule.weave.extension.api.component.structure.WeaveProjectStructure
import org.mule.weave.extension.api.extension.action.WeaveCodeActionProvider
import org.mule.weave.extension.api.extension.annotations.WeaveAnnotationProcessorBinding
import org.mule.weave.extension.api.extension.command.WeaveCommand
import org.mule.weave.extension.api.extension.validation.WeaveValidator
import org.mule.weave.lsp.project.components.DefaultProjectFolderFactory
import org.mule.weave.lsp.project.components.ProjectFolderFactory
import org.mule.weave.lsp.project.components.ProjectStructureHelper
import org.mule.weave.lsp.project.components.SampleDataComponent
import org.mule.weave.lsp.services.ToolingService
import org.mule.weave.lsp.utils.InternalEventBus
import org.mule.weave.v2.editor.VirtualFileSystem

import java.io.File

/**
 * A Project Kind is the trait that allows us to support multiple kind of projects with different:
 *  - builds
 *  - dependency management
 *  - folder structure
 */
trait ProjectKind {

  /**
   * The name of the kind i.e maven, bat, simple ...
   *
   * @return
   */
  def name(): String

  /**
   * Setups the project kind. Download any tool required for this kind of project
   */
  def initialized(): Unit

  def initialize(): Unit

  def shutdown(): Unit

  def eventBus(): InternalEventBus;

  def vfs(): VirtualFileSystem

  def toolingService[T <: ToolingService](toolingService: Class[T]): T

  /**
   * Returns the Project structure with all the modules and it source folders
   *
   * @return
   */
  def structure(): WeaveProjectStructure

  /**
   * Returns whether a uri is a file of the given kind project
   *
   * @param uri the uri
   * @return
   */
  def isProjectFile(uri: String): Boolean = {
    ProjectStructureHelper.isAProjectFile(uri, structure())
  }

  /**
   * The Dependency Manager handles all the dependencies for this kind of projects
   *
   * @return
   */
  def dependencyManager(): WeaveDependencyComponent

  /**
   * Handles the build, deploy of this project
   *
   * @return
   */
  def buildManager(): WeaveBuildComponent

  /**
   * Handles Scenario Provider for sample data
   *
   * @return The Scenarios For sample data
   */
  def sampleDataManager(): SampleDataComponent

  /**
   * Handles the metadata for the files.
   *
   * @return The Metadata Provider if this Project can provide any
   */
  def metadataProvider(): WeaveMetadataProviderComponent

  def textDocumentLocator(): WeaveTextDocumentLocator

  /**
   * Returns any additional change that needs to be made when a new file is created
   *
   * @param folder the folder
   * @param name   the file name
   * @return
   */
  def newFile(folder: File, name: String): Array[Either[TextDocumentEdit, ResourceOperation]] = Array.empty

  /**
   * Returns the folder factory for the given project
   *
   * @return
   */
  def projectFolderFactory(): ProjectFolderFactory = DefaultProjectFolderFactory

  /**
   * Get a custom set of CodeActionProvider according to the project kind
   *
   * @return A custom set of CodeActionProvider according to the project kind
   */
  def customCodeActions(): Array[WeaveCodeActionProvider] = Array.empty

  /**
   * Returns a custom set of WeaveCommands according to the project kind
   */
  def customCommands(): Array[WeaveCommand] = Array.empty

  /**
   * Returns a custom set of annotations processors to be used.
   */
  def customAnnotations(): Array[WeaveAnnotationProcessorBinding] = Array.empty

  /**
   * Returns a custom set of annotations processors to be used.
   */
  def customValidators(): Array[WeaveValidator] = Array.empty

  /**
   * Get the DataWeave version for the current project kind. For maven projects
   * it's obtained first from the POM, otherwise it defaults to the dependency manager language level.
   *
   * @return The current dataweave version in major.minor format (2.4, 2.5).
   */
  def getWeaveVersion(): String = {
    val version = dependencyManager().dwVersion().split("\\.")
    s"${version(0)}.${version(1)}"
  }
}
